<?php

use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Volt\Component;

new #[Layout('components.layouts.auth')] class extends Component {
    #[Validate('required|string|email')]
    public string $email = '';

    #[Validate('required|string')]
    public string $password = '';

    public bool $remember = false;

    /**
     * Handle an incoming authentication request.
     */
    public function login(): void
    {
        $this->validate();

        $this->ensureIsNotRateLimited();

        if (! Auth::attempt(['email' => $this->email, 'password' => $this->password], $this->remember)) {
            RateLimiter::hit($this->throttleKey());

            throw ValidationException::withMessages([
                'email' => __('auth.failed'),
            ]);
        }

        RateLimiter::clear($this->throttleKey());
        Session::regenerate();

        $this->redirectIntended(default: route('dashboard', absolute: false), navigate: true);
    }

    /**
     * Ensure the authentication request is not rate limited.
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'email' => __('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the authentication rate limiting throttle key.
     */
    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->email).'|'.request()->ip());
    }
}; ?>



<div class="custom-login-container">
    <style>
    /* Desactivar barra de progreso de Livewire */
    [wire\:loading-bar] { 
        display: none !important;
    }

    /* Sobrescribir todos los estilos restrictivos del layout */
    .bg-background,
    .flex.min-h-svh,
    .flex.w-full.max-w-sm {
        max-width: none !important;
        width: 100% !important;
    }

    body > div > div,
    body > div > div > div {
        max-width: none !important;
        width: 100% !important;
        gap: 0 !important;
    }

    .custom-login-container {
        background: white;
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        overflow: hidden;
        max-width: 900px !important;
        width: 100% !important;
        display: grid;
        grid-template-columns: 1fr 1fr;
        margin: 0 auto;
    }

    .custom-login-left {
        background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 100%);
        padding: 60px 40px;
        color: white;
        display: flex;
        flex-direction: column;
        justify-content: center;
        align-items: center;
        text-align: center;
    }

    .custom-login-left h1 {
        font-size: 2.5em;
        margin-bottom: 20px;
        font-weight: 700;
    }

    .custom-login-left p {
        font-size: 1.1em;
        opacity: 0.9;
        line-height: 1.6;
    }

    .custom-welcome-icon {
        width: 120px;
        height: 120px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 30px;
        font-size: 50px;
    }

    .custom-login-right {
        padding: 60px 40px;
        display: flex;
        flex-direction: column;
        justify-content: center;
    }

    .custom-login-right h2 {
        color: #1e3a8a;
        font-size: 2em;
        margin-bottom: 10px;
    }

    .custom-login-right .custom-subtitle {
        color: #64748b;
        margin-bottom: 40px;
        font-size: 0.95em;
    }

    .custom-form-group {
        margin-bottom: 25px;
    }

    .custom-form-group label {
        display: block;
        color: #334155;
        font-weight: 600;
        margin-bottom: 8px;
        font-size: 0.9em;
    }

    .custom-input-wrapper {
        position: relative;
    }

    .custom-input-wrapper svg {
        position: absolute;
        left: 15px;
        top: 50%;
        transform: translateY(-50%);
        width: 20px;
        height: 20px;
        color: #64748b;
    }

    .custom-form-group input {
        width: 100%;
        padding: 14px 14px 14px 45px;
        border: 2px solid #e2e8f0;
        border-radius: 10px;
        font-size: 0.95em;
        transition: all 0.3s ease;
        outline: none;
    }

    .custom-form-group input:focus {
        border-color: #f97316;
        box-shadow: 0 0 0 3px rgba(249, 115, 22, 0.1);
    }

    .custom-form-options {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 30px;
        font-size: 0.9em;
    }

    .custom-remember-me {
        display: flex;
        align-items: center;
        gap: 8px;
        color: #64748b;
    }

    .custom-remember-me input[type="checkbox"] {
        width: 18px;
        height: 18px;
        cursor: pointer;
        accent-color: #f97316;
    }

    .custom-forgot-password {
        color: #f97316;
        text-decoration: none;
        font-weight: 600;
        transition: color 0.3s ease;
    }

    .custom-forgot-password:hover {
        color: #ea580c;
    }

    .custom-login-button {
        width: 100%;
        padding: 16px;
        background: linear-gradient(135deg, #f97316 0%, #fb923c 100%);
        color: white;
        border: none;
        border-radius: 10px;
        font-size: 1em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(249, 115, 22, 0.3);
    }

    .custom-login-button:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(249, 115, 22, 0.4);
    }

    .custom-login-button:active {
        transform: translateY(0);
    }

    .custom-signup-link {
        text-align: center;
        margin-top: 25px;
        color: #64748b;
        font-size: 0.9em;
    }

    .custom-signup-link a {
        color: #3b82f6;
        text-decoration: none;
        font-weight: 600;
        transition: color 0.3s ease;
    }

    .custom-signup-link a:hover {
        color: #1e3a8a;
    }

    .custom-error-message {
        color: #dc2626;
        font-size: 0.85em;
        margin-top: 5px;
    }

    .custom-logo {
        width: 200px;
        height: auto;
        margin-bottom: 30px;
    }

    @media (max-width: 768px) {
        .custom-login-container {
            grid-template-columns: 1fr;
        }

        .custom-login-left {
            padding: 40px 30px;
        }

        .custom-login-left h1 {
            font-size: 2em;
        }

        .custom-login-right {
            padding: 40px 30px;
        }
    }
</style>
    <div class="custom-login-left">
        <img src="{{ asset('imagenes/logo_enkelt.avif') }}" alt="Logo" class="custom-logo">
        <h1>¡Bienvenido!</h1>
        <p>Accede a tu cuenta para disfrutar de todas nuestras funcionalidades y servicios exclusivos.</p>
    </div>

    <div class="custom-login-right">
        <h2>Iniciar Sesión</h2>
        <p class="custom-subtitle">Ingresa tus credenciales para continuar</p>

        <!-- Session Status -->
        <x-auth-session-status class="mb-4" :status="session('status')" />

        <form wire:submit.prevent="login">
            <!-- Email -->
            <div class="custom-form-group">
                <label for="email">{{ __('E-mail') }}</label>
                <div class="custom-input-wrapper">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                    </svg>
                    <input
                        type="email"
                        id="email"
                        wire:model.debounce.500ms="email"
                        required
                        autofocus
                        autocomplete="email"
                    >
                </div>
                @error('email')
                    <div class="custom-error-message">{{ $message }}</div>
                @enderror
            </div>

            <!-- Password -->
            <div class="custom-form-group">
                <label for="password">{{ __('Contraseña') }}</label>
                <div class="custom-input-wrapper">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                    </svg>
                    <input 
                        type="password" 
                        id="password" 
                        wire:model.debounce.500ms="password"
                        placeholder="••••••••" 
                        required
                        autocomplete="current-password"
                    >
                </div>
                @error('password')
                    <div class="custom-error-message">{{ $message }}</div>
                @enderror
            </div>

            <div class="custom-form-options">
                <label class="custom-remember-me">
                    <input type="checkbox" id="remember" wire:model="remember">
                    <span>{{ __('Recordar') }}</span>
                </label>
                @if (Route::has('password.request'))
                    <a href="{{ route('password.request') }}" class="custom-forgot-password" wire:navigate>
                        {{ __('¿Olvidaste tu contraseña?') }}
                    </a>
                @endif
            </div>

            <button type="submit" class="custom-login-button" wire:loading.attr="disabled">
                <span wire:loading.remove>{{ __('Iniciar sesión') }}</span>
                <span wire:loading>Iniciando sesión...</span>
            </button>
        </form>
    </div>
</div>
