<?php

namespace App\Livewire\Ventas\Cotizaciones;

class CotizacionesBaseLogic
{
    private const DEFAULT_IVA = '16';
    private const DEFAULT_RETENCION_IVA = '10.67';
    public static function calcularImpuestos($cantidad, $precioAntesIva, $descuento = 0)
    {
        $precision = 8;
        $cantidad = (string) $cantidad;
        $precio = (string) $precioAntesIva;
        $descuento = (string) $descuento;
        
    $importe = bcmul($cantidad, $precio, $precision);
    $subtotal = bcsub($importe, $descuento, $precision);

    $ivaFactor = bcdiv(self::DEFAULT_IVA, '100', $precision);
    $retIvaFactor = bcdiv(self::DEFAULT_RETENCION_IVA, '100', $precision);

    $ivatrasladado = bcmul($subtotal, $ivaFactor, $precision);
    $ivaretenido = bcmul($subtotal, $retIvaFactor, $precision);
        $iepstrasladado = '0.00';
        $isrretenido = '0.00';
    $precioConIva = bcadd($precio, bcmul($precio, $ivaFactor, $precision), $precision);
        
        return [
            'importe' => $importe,
            'subtotal' => $subtotal,
            'ivatrasladado' => $ivatrasladado,
            'ivaretenido' => $ivaretenido,
            'iepstrasladado' => $iepstrasladado,
            'isrretenido' => $isrretenido,
            'precioConIva' => $precioConIva,
        ];
    }

    public static function sincronizarCalculosModal($partidaEditando)
    {
        $precision = 8;
        $cantidad = (string) max(1, (int) ($partidaEditando['cantidad'] ?? 1));
        $precioUnitario = (string) ($partidaEditando['precio_unitario'] ?? '0');
        $descuentoTotal = (string) ($partidaEditando['descuento_total'] ?? '0');
        $ivaImporte = (string) ($partidaEditando['iva_importe'] ?? '0');
        $iepsImporte = (string) ($partidaEditando['ieps_importe'] ?? '0');
        $retencionIvaImporte = (string) ($partidaEditando['retencion_iva_importe'] ?? '0');
        $retencionIsrImporte = (string) ($partidaEditando['retencion_isr_importe'] ?? '0');

        $importe = bcmul($cantidad, $precioUnitario, $precision);
        $subtotal = bcsub($importe, $descuentoTotal, $precision);
        $sumImpuestos = bcadd($ivaImporte, $iepsImporte, $precision);
        $sumRetenciones = bcadd($retencionIvaImporte, $retencionIsrImporte, $precision);
        $total = bcsub(bcadd($subtotal, $sumImpuestos, $precision), $sumRetenciones, $precision);

        return [
            'importe' => (float) self::bcround($importe, 2),
            'subtotal' => (float) self::bcround($subtotal, 2),
            'iva_trasladado' => (float) self::bcround($ivaImporte, 2),
            'ieps_trasladado' => (float) self::bcround($iepsImporte, 2),
            'iva_retenido' => (float) self::bcround($retencionIvaImporte, 2),
            'isr_retenido' => (float) self::bcround($retencionIsrImporte, 2),
            'total' => (float) self::bcround($total, 2),
            'iva_porcentaje' => (float) ($partidaEditando['iva_porcentaje'] ?? self::DEFAULT_IVA),
            'iva_retenido_porcentaje' => (float) ($partidaEditando['retencion_iva_porcentaje'] ?? self::DEFAULT_RETENCION_IVA),
            'ieps_porcentaje' => (float) ($partidaEditando['ieps_porcentaje'] ?? 0),
            'isr_retenido_porcentaje' => (float) ($partidaEditando['retencion_isr_porcentaje'] ?? 0),
        ];
    }

    private static function bcround(string $number, int $precision = 2): string
    {
        $scale = max(0, $precision) + 1;
        $increment = '0.' . str_repeat('0', max(0, $precision)) . '5';
        $comp = bccomp($number, '0', $scale);
        $adjusted = $comp >= 0
            ? bcadd($number, $increment, $scale)
            : bcsub($number, $increment, $scale);
        return bcadd($adjusted, '0', max(0, $precision));
    }
    
    public static function calcularTotalProducto($cantidad, $precioAntesIva, $descuento = 0)
    {
        $precision = 8;
        $impuestos = self::calcularImpuestos($cantidad, $precioAntesIva, $descuento);
        
        $total = bcadd($impuestos['subtotal'], $impuestos['ivatrasladado'], $precision);
        $total = bcadd($total, $impuestos['iepstrasladado'], $precision);
        $total = bcsub($total, $impuestos['ivaretenido'], $precision);
        $total = bcsub($total, $impuestos['isrretenido'], $precision);
        
        return array_merge($impuestos, ['total' => $total]);
    }
    
    public static function calcularTotalesCotizacion($productosAgregados)
    {
        $precision = 8;
        $totales = [
            'importetotal' => '0.00',
            'descuentototal' => '0.00',
            'subtotaltotal' => '0.00',
            'retivatotal' => '0.00',
            'retisrtotal' => '0.00',
            'iepstotal' => '0.00',
            'impuestototal' => '0.00',
            'totalfinal' => '0.00',
        ];
        
        foreach ($productosAgregados as $producto) {
            $totales['importetotal'] = bcadd($totales['importetotal'], $producto['importe'], $precision);
            $totales['descuentototal'] = bcadd($totales['descuentototal'], $producto['descuento'], $precision);
            $totales['subtotaltotal'] = bcadd($totales['subtotaltotal'], $producto['subtotal'], $precision);
            $totales['retivatotal'] = bcadd($totales['retivatotal'], $producto['iva_retenido'], $precision);
            $totales['retisrtotal'] = bcadd($totales['retisrtotal'], $producto['isr_retenido'], $precision);
            $totales['iepstotal'] = bcadd($totales['iepstotal'], $producto['ieps_trasladado'], $precision);
            $totales['impuestototal'] = bcadd($totales['impuestototal'], $producto['iva_trasladado'], $precision);
        }
        
        $totales['totalfinal'] = $totales['subtotaltotal'];
        $totales['totalfinal'] = bcadd($totales['totalfinal'], $totales['impuestototal'], $precision);
        $totales['totalfinal'] = bcadd($totales['totalfinal'], $totales['iepstotal'], $precision);
        $totales['totalfinal'] = bcsub($totales['totalfinal'], $totales['retivatotal'], $precision);
        $totales['totalfinal'] = bcsub($totales['totalfinal'], $totales['retisrtotal'], $precision);
        
        return $totales;
    }
    
    public static function validarCalculos($productosAgregados, $totalFinal)
    {
        $precision = 8;
        $totalCalculado = '0.00';
        
        foreach ($productosAgregados as $producto) {
            $totalCalculado = bcadd($totalCalculado, $producto['total'], $precision);
        }
        
        return bccomp($totalCalculado, $totalFinal, 4) === 0;
    }
    
    public static function numeroALetras($numero)
    {
        try {
            $numeroFloat = floatval($numero);
            $entero = intval($numeroFloat);
            $decimales = intval(($numeroFloat - $entero) * 100);
            
            if ($entero == 0) {
                $letras = "CERO";
            } elseif ($entero == 1) {
                $letras = "UN";
            } else {
                $letras = "CANTIDAD: $" . number_format($entero, 0);
            }
            
            return "(" . strtoupper($letras) . ") PESOS " . sprintf("%02d", $decimales);
            
        } catch (\Exception $e) {
            return "CERO PESOS";
        }
    }
    
    public static function formatearNumero($numero, $decimales = null)
    {
        $valor = floatval($numero);
        
        if ($decimales === null) {
            $formateado = rtrim(rtrim(number_format($valor, 8, '.', ','), '0'), '.');
            if (strpos($formateado, '.') === false) {
                $formateado .= '.00';
            }
            return $formateado;
        }
        
        return number_format($valor, $decimales, '.', ',');
    }
    
    public static function formatearMoneda($numero, $decimales = null)
    {
        return '$ ' . self::formatearNumero($numero, $decimales);
    }
    
    public static function crearProductoParaCotizacion($datos)
    {
        $calculo = self::calcularTotalProducto(
            $datos['cantidad'],
            $datos['precio'],
            $datos['descuento']
        );
        
        return [
            'id' => uniqid(),
            'producto_id' => $datos['productoid'],
            'clave' => $datos['clave'],
            'descripcion' => $datos['descripcion'],
            'cantidad' => $datos['cantidad'],
            'precio_unitario' => $datos['precio'],
            'importe' => $calculo['importe'],
            'descuento' => $datos['descuento'],
            'subtotal' => $calculo['subtotal'],
            'iva_trasladado' => $calculo['ivatrasladado'],
            'iva_retenido' => $calculo['ivaretenido'],
            'ieps_trasladado' => $calculo['iepstrasladado'],
            'isr_retenido' => $calculo['isrretenido'],
            'total' => $calculo['total'],
            'descripcion_adicional' => $datos['descripcion_adicional'] ?? '',
        ];
    }

    public static function getTotalPartidas($productosAgregados)
    {
        return count($productosAgregados);
    }

    public static function limpiarCamposProducto()
    {
        return [
            'productoid' => null,
            'productosearch' => '',
            'cantidad' => 1,
            'precioventaantesiva' => 0,
            'precioventaconiva' => 0,
            'descuento' => 0,
            'descripcion' => '',
            'descripcionadicional' => '',
            'claveproductoservicio' => '',
            'descripcionclaveproductoservicio' => '',
            'claveunidadmedida' => '',
            'descripcionunidadmedida' => '',
            'ivatrasladado' => 0,
            'ivaretenido' => 0,
            'iepstrasladado' => 0,
            'isrretenido' => 0,
        ];
    }

    public static function construirDireccionCliente($cliente)
    {
        $direccion = [];
        
        if (!empty($cliente['calle'])) {
            $direccion[] = $cliente['calle'];
        }
        if (!empty($cliente['noexterior'])) {
            $direccion[] = 'No. ' . $cliente['noexterior'];
        }
        if (!empty($cliente['colonia'])) {
            $direccion[] = $cliente['colonia'];
        }
        if (!empty($cliente['ciudad'])) {
            $direccion[] = $cliente['ciudad'];
        }
        
        return implode(', ', $direccion);
    }
}
