<?php

namespace App\Livewire\Reportes;

use Livewire\Attributes\Title;
use Livewire\Component;
use App\Models\Sucursal;
use App\Models\Familia;
use App\Models\Usuario;
use App\Models\VentasDetalle;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\Reportes\VentasFamiliaProductoExport;
use Mpdf\Mpdf;

#[Title('Ventas por Familia/Producto')]
class VentasFamiliaProducto extends Component
{
    // Filtros
    public $fechainicial;
    public $fechafinal;
    public $idsucursal = '0';
    public $tipoVenta = 'familia';
    public $idfamilia = '0';
    public $idusuario = '0';

    // Configuración de turnos
    public $usarTurnos = false; // Si true, usa turnos; si false, usa fechas
    public $tipoSeleccionTurnos = 'rango'; // 'rango', 'individual', 'todos'
    public $turnoInicial = '';
    public $turnoFinal = '';
    public $turnosIndividuales = ''; // Separados por comas
    public $considerarFechas = false; // Para cuando se usan turnos pero también se consideran fechas
    public $turnosLabel = '';

    // Datos para selects
    public $sucursales = [];
    public $familias = [];
    public $usuarios = [];

    // Datos de la tabla
    public $ventasPorFamilia = [];

    // Totales
    public $totalCantidad = 0;
    public $totalCosto = 0;
    public $totalSubtotal = 0;
    public $totalIva = 0;
    public $totalTotal = 0;
    public $totalUtilidad = 0;

    // Modal estados
    public $modalTurnosOpen = false;
    public $modalExportarOpen = false;

    public function mount()
    {
        // Inicializar fechas con el mes actual
        $this->fechainicial = now()->startOfMonth()->format('Y-m-d');
        $this->fechafinal = now()->format('Y-m-d');

        // Cargar datos iniciales
        $this->cargarDatos();
    }

    public function cargarDatos()
    {

        $this->sucursales = Sucursal::orderBy('razon_social')
            ->pluck('razon_social', 'idsucursal')
            ->toArray();

        $this->familias = Familia::orderBy('nombrefamilia')
            ->pluck('nombrefamilia', 'idfamilia')
            ->toArray();

        $this->usuarios = Usuario::orderBy('nombre')
            ->pluck('nombre', 'idusuario')
            ->toArray();
    }

    public function updatedIdSucursal()
    {

        if ($this->idsucursal > 0) {
            $this->usuarios = Usuario::orderBy('nombre')
                ->where('idsucursal', $this->idsucursal)
                ->pluck('nombre', 'iduser')
                ->toArray();
        } else {
            $this->usuarios = Usuario::orderBy('nombre')
                ->pluck('nombre', 'idusuario')
                ->toArray();
        }
    }

    public function consultar()
    {
        $this->ventasPorFamilia = [];
        $this->resetearTotales();

        // Construir la consulta base usando el modelo
        $query = VentasDetalle::query();
        // Determinar si es por familia o producto
        if ($this->tipoVenta == 'familia') {
            $query->join('cat_familias as f', 'f.idfamily', '=', 'ventasdetalle.idfamilia')
                ->select(
                    'f.nombrefamilia as nombre',
                    DB::raw('SUM(ventasdetalle.cantidad) as cantidad'),
                    DB::raw('SUM(ventasdetalle.costo_venta) as costo'),
                    DB::raw('SUM(ventasdetalle.subtotal) as subtotal'),
                    DB::raw('SUM(ventasdetalle.iva) as iva'),
                    DB::raw('SUM(ventasdetalle.total) as total'),
                    DB::raw('SUM(ventasdetalle.utilidad) as utilidad')
                )
                ->groupBy('ventasdetalle.idfamilia', 'f.nombrefamilia');
        } else {
            // ventas por producto
            $query->join('cat_productos as p', 'p.idproduct', '=', 'ventasdetalle.idproducto')
                ->select(
                    'p.descripcion as nombre',
                    DB::raw('SUM(ventasdetalle.cantidad) as cantidad'),
                    DB::raw('SUM(ventasdetalle.costo_venta) as costo'),
                    DB::raw('SUM(ventasdetalle.subtotal) as subtotal'),
                    DB::raw('SUM(ventasdetalle.iva) as iva'),
                    DB::raw('SUM(ventasdetalle.total) as total'),
                    DB::raw('SUM(ventasdetalle.utilidad) as utilidad')
                )
                ->groupBy('ventasdetalle.idproducto', 'p.descripcion');
        }

        // Aplicar filtros opcionales
        $query->when($this->idfamilia > 0, function ($q) {
            return $q->where('ventasdetalle.idfamilia', $this->idfamilia);
        })
            ->when($this->idusuario > 0, function ($q) {
                if ($this->idsucursal > 0) {
                    //Se filtra por idusuario
                    return $q->where('ventasdetalle.idusuario', $this->idusuario);
                }
                return $q->where('ventasdetalle.idusuario_desk', $this->idusuario);
            })
            ->when($this->idsucursal > 0, function ($q) {
                return $q->where('ventasdetalle.idsucursal', $this->idsucursal);
            });

        // Aplicar filtros de fechas y/o turnos según la configuración
        // Caso 1: No considerar turnos O considerar fechas junto con turnos
        if (!$this->usarTurnos || $this->considerarFechas) {
            $query->whereBetween('ventasdetalle.fechahora', [
                $this->fechainicial . ' 00:00:00',
                $this->fechafinal . ' 23:59:59'
            ]);
        }

        // Caso 2: Usar turnos
        if ($this->usarTurnos) {
            if ($this->tipoSeleccionTurnos === 'rango' && !empty($this->turnoInicial) && !empty($this->turnoFinal)) {
                // Rango de turnos
                $query->whereBetween('ventasdetalle.idturno', [(int)$this->turnoInicial, (int)$this->turnoFinal]);
            } elseif ($this->tipoSeleccionTurnos === 'individual' && !empty($this->turnosIndividuales)) {
                // Turnos individuales (precisos)
                $turnos = collect(explode(',', $this->turnosIndividuales))
                    ->map(fn($turno) => (int)trim($turno))
                    ->filter()
                    ->toArray();

                if (!empty($turnos)) {
                    $query->whereIn('ventasdetalle.idturno', $turnos);
                }
            }
            // Si es 'todos', no aplicamos ningún filtro de turno
        }

        // Obtener los resultados y ordenarlos después
        $query = $query->orderBy('total', 'desc');

        $resultados = $query->get();

        // Convertir a array de objetos y calcular totales
        $this->ventasPorFamilia = $resultados->map(function ($item) {
            $this->totalCantidad += $item->cantidad ?? 0;
            $this->totalCosto += $item->costo ?? 0;
            $this->totalSubtotal += $item->subtotal ?? 0;
            $this->totalIva += $item->iva ?? 0;
            $this->totalTotal += $item->total ?? 0;
            $this->totalUtilidad += $item->utilidad ?? 0;

            return (object) [
                'nombre' => $item->nombre,
                'cantidad' => $item->cantidad ?? 0,
                'costo' => $item->costo ?? 0,
                'subtotal' => $item->subtotal ?? 0,
                'iva' => $item->iva ?? 0,
                'total' => $item->total ?? 0,
                'utilidad' => $item->utilidad ?? 0,
            ];
        })->toArray();
    }

    public function seleccionarTurnos()
    {
        // Validaciones según el tipo de selección
        if ($this->tipoSeleccionTurnos === 'rango') {
            $this->validate([
                'turnoInicial' => 'required|numeric|min:1',
                'turnoFinal' => 'required|numeric|min:1|gte:turnoInicial',
            ], [
                'turnoInicial.required' => 'El turno inicial es requerido',
                'turnoInicial.numeric' => 'El turno inicial debe ser un número',
                'turnoInicial.min' => 'El turno inicial debe ser mayor a 0',
                'turnoFinal.required' => 'El turno final es requerido',
                'turnoFinal.numeric' => 'El turno final debe ser un número',
                'turnoFinal.min' => 'El turno final debe ser mayor a 0',
                'turnoFinal.gte' => 'El turno final debe ser mayor o igual al turno inicial',
            ]);
        } elseif ($this->tipoSeleccionTurnos === 'individual') {
            // Limpiar el input antes de validar
            $this->turnosIndividuales = $this->limpiarTurnosInput($this->turnosIndividuales);

            $this->validate([
                'turnosIndividuales' => ['required', 'regex:/^[0-9]+(,[0-9]+)*$/'],
            ], [
                'turnosIndividuales.required' => 'Debes ingresar al menos un turno',
                'turnosIndividuales.regex' => 'El formato debe ser números separados por comas (Ej: 1,5,10)',
            ]);
        }

        // Activar el uso de turnos
        $this->usarTurnos = true;

        $this->modalTurnosOpen = false;

        $this->turnosLabel = $this->getTurnosLabel();
    }

    public function limpiarTurnosInput($input)
    {
        $cleaned = preg_replace('/[^0-9,]/', '', $input);

        // Remover comas múltiples consecutivas
        $cleaned = preg_replace('/,+/', ',', $cleaned);

        // Remover comas al inicio y al final
        $cleaned = trim($cleaned, ',');

        return $cleaned;
    }

    private function getTurnosLabel()
    {
        if (!$this->usarTurnos) {
            return 'Sin filtro de turnos';
        }

        if ($this->tipoSeleccionTurnos === 'todos') {
            $label = 'Todos los turnos';
        } elseif ($this->tipoSeleccionTurnos === 'rango') {
            $label = "Turnos del {$this->turnoInicial} al {$this->turnoFinal}";
        } else {
            // individual
            $turnos = explode(',', $this->turnosIndividuales);
            $count = count($turnos);
            if ($count <= 3) {
                $label = "Turnos: " . $this->turnosIndividuales;
            } else {
                $primeros = array_slice($turnos, 0, 3);
                $label = "Turnos: " . implode(', ', $primeros) . "... (+{$count} turnos)";
            }
        }

        if ($this->considerarFechas) {
            $label .= ' (con filtro de fechas)';
        }

        return $label;
    }

    public function limpiarTurnos()
    {
        $this->usarTurnos = false;
        $this->tipoSeleccionTurnos = 'rango';
        $this->turnoInicial = '';
        $this->turnoFinal = '';
        $this->turnosIndividuales = '';
        $this->considerarFechas = false;
        $this->turnosLabel = '';
    }

    public function resetearTotales()
    {
        $this->totalCantidad = 0;
        $this->totalCosto = 0;
        $this->totalSubtotal = 0;
        $this->totalIva = 0;
        $this->totalTotal = 0;
        $this->totalUtilidad = 0;
    }

    public function exportar(string $tipo)
    {
        // Validar que haya datos
        if (empty($this->ventasPorFamilia)) {
            session()->flash('error', 'No hay datos para exportar. Por favor, ejecuta primero la consulta.');
            $this->modalExportarOpen = false;
            return;
        }

        $filename = 'ventas_' . ($this->tipoVenta === 'familia' ? 'familia' : 'producto') . '_' . date('Ymd_His');

        if ($tipo === 'excel') {
            return $this->exportarExcel($filename);
        } elseif ($tipo === 'pdf') {
            return $this->exportarPDF($filename);
        }

        $this->modalExportarOpen = false;
    }

    private function exportarExcel($filename)
    {
        $totales = [
            'cantidad' => $this->totalCantidad,
            'costo' => $this->totalCosto,
            'subtotal' => $this->totalSubtotal,
            'iva' => $this->totalIva,
            'total' => $this->totalTotal,
            'utilidad' => $this->totalUtilidad,
        ];

        return Excel::download(
            new VentasFamiliaProductoExport($this->ventasPorFamilia, $this->tipoVenta, $totales),
            $filename . '.xlsx'
        );
    }
    private function exportarPDF($filename)
    {
        // Preparar datos para la vista
        $data = collect($this->ventasPorFamilia)->map(function ($item) {
            return (object) $item;
        });

        $totales = [
            'cantidad' => $this->totalCantidad,
            'costo' => $this->totalCosto,
            'subtotal' => $this->totalSubtotal,
            'iva' => $this->totalIva,
            'total' => $this->totalTotal,
            'utilidad' => $this->totalUtilidad,
        ];
        // Obtener nombres legibles para filtros
        $sucursal = $this->idsucursal > 0 ? ($this->sucursales[$this->idsucursal] ?? null) : null;
        $familia = $this->idfamilia > 0 ? ($this->familias[$this->idfamilia] ?? null) : null;
        $usuario = $this->idusuario > 0 ? ($this->usuarios[$this->idusuario] ?? null) : null;
        $turnosInfo = $this->usarTurnos ? $this->turnosLabel : null;

        $html = view('exports.ventas-familia-producto', [
            'data' => $data,
            'tipoVenta' => $this->tipoVenta,
            'totales' => $totales,
            'fechainicial' => $this->fechainicial,
            'fechafinal' => $this->fechafinal,
            'sucursal' => $sucursal,
            'familia' => $familia,
            'usuario' => $usuario,
            'turnosInfo' => $turnosInfo,
        ])->render();

        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L', // Formato horizontal para más columnas
            'margin_left' => 10,
            'margin_right' => 10,
            'margin_top' => 10,
            'margin_bottom' => 20,
        ]);

        $mpdf->WriteHTML($html);

        return response()->streamDownload(function () use ($mpdf) {
            $mpdf->Output('', 'D');
        }, $filename . '.pdf');
    }

    public function render()
    {
        return view('livewire.reportes.ventas-familia-producto.index');
    }
}
