<?php

namespace App\Livewire\Reportes\LoMasMenosVendido;

use Livewire\Attributes\Title;
use Livewire\Component;
use App\Models\Producto;
use App\Models\Sucursal;
use App\Models\Familia;
use App\Models\Marca;
use App\Models\Usuario;
use App\Models\VentasDetalle;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

#[Title('Reporte de MasMenosVendido')]
class Index extends Component
{
    public $orden = 'masvendidos';
    public $fechainicio;
    public $fechafinal;
    public $turno = '';
    public $sucursal = '';
    public $usuario = '';
    public $familia = '';
    public $marca = '';
    public $filas = 25;

    public function mount()
    {
        $fechaInfo = DB::connection('tenant')->table('ing_ventasdetalle01')
            ->join('ing_ventasgeneral01', 'ing_ventasdetalle01.idventageneral', '=', 'ing_ventasgeneral01.idweb')
            ->selectRaw('MIN(ing_ventasgeneral01.fechaventa) as primera, MAX(ing_ventasgeneral01.fechaventa) as ultima')
            ->first();
        
        if ($fechaInfo && $fechaInfo->ultima) {
            $ultimaFecha = \Carbon\Carbon::parse($fechaInfo->ultima);
            $this->fechainicio = $ultimaFecha->startOfMonth()->format('Y-m-d');
            $this->fechafinal = $ultimaFecha->endOfMonth()->format('Y-m-d');
        } else {
            $this->fechainicio = now()->startOfMonth()->format('Y-m-d');
            $this->fechafinal = now()->endOfMonth()->format('Y-m-d');
        }
    }

    public function consultar()
    {
        $this->render();
    }

    public function render()
    {
        $query = VentasDetalle::query()
            ->join('ing_ventasgeneral01', 'ing_ventasdetalle01.idventageneral', '=', 'ing_ventasgeneral01.idweb')
            ->select([
                'ing_ventasdetalle01.idproducto',
                DB::raw('MAX(ing_ventasdetalle01.descripcion) as descripcion'),
                DB::raw('MAX(ing_ventasdetalle01.idfamilia) as idfamilia'),
                DB::raw('SUM(ing_ventasdetalle01.cantidad) as cantidad_vendida'),
                DB::raw('SUM(ing_ventasdetalle01.total) as total_vendido'),
                DB::raw('SUM(ing_ventasdetalle01.costo_venta * ing_ventasdetalle01.cantidad) as costo'),
                DB::raw('SUM(ing_ventasdetalle01.utilidad) as utilidad')
            ]);

        if ($this->fechainicio && $this->fechafinal) {
            $query->whereBetween('ing_ventasgeneral01.fechaventa', [
                $this->fechainicio,
                $this->fechafinal
            ]);
        }

        if ($this->sucursal) {
            $query->where('ing_ventasdetalle01.idsucursal', (int) $this->sucursal);
        }

        if ($this->familia) {
            $query->where('ing_ventasdetalle01.idfamilia', (int) $this->familia);
        }

        $query->groupBy('ing_ventasdetalle01.idproducto');
        
        $direccionOrden = ($this->orden == 'masvendidos') ? 'desc' : 'asc';
        $query->orderBy(DB::raw('SUM(ing_ventasdetalle01.cantidad)'), $direccionOrden)
              ->limit((int) $this->filas);

        $productos = $query->get();

        $idsProductos = $productos->pluck('idproducto')->toArray();
        if (!empty($idsProductos)) {
            $existencias = DB::table('mov_inv_existencias')
                ->select('idproducto', DB::raw('SUM(existencia) as existencia_total'))
                ->whereIn('idproducto', $idsProductos)
                ->groupBy('idproducto')
                ->pluck('existencia_total', 'idproducto');

            foreach ($productos as $producto) {
                $producto->existencia_actual = $existencias[$producto->idproducto] ?? 0;
            }
        }

        if ($productos->isNotEmpty()) {
            $familias = Familia::whereIn('idfamily', $productos->pluck('idfamilia')->filter()->unique())->get()->keyBy('idfamily');
            $productosModels = Producto::whereIn('idproduct', $productos->pluck('idproducto'))->with('marca')->get()->keyBy('idproduct');

            foreach ($productos as $producto) {
                $producto->familia = $familias[$producto->idfamilia] ?? null;
                $productoModel = $productosModels[$producto->idproducto] ?? null;
                $producto->marca = $productoModel?->marca ?? null;
            }
        }

        $totales = [
            'cantidadvendida' => $productos->sum('cantidad_vendida'),
            'totalvendido'    => $productos->sum('total_vendido'),
            'costo'           => $productos->sum('costo'),
            'utilidad'        => $productos->sum('utilidad'),
        ];

        return view('livewire.reportes.lo_mas_menos_vendido.index', [
            'productos' => $productos,
            'totales' => $totales,
            'sucursales' => Sucursal::orderBy('razon_social')->get(),
            'familias' => Familia::orderBy('nombrefamilia')->get(),
            'marcas' => Marca::orderBy('marca')->get(),
            'usuarios' => Usuario::orderBy('nombre')->get(),
        ]);
    }

}
