<?php

namespace App\Livewire\Reportes\CortesCaja;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Title;
use Livewire\Component;

#[Title('Reporte Concentrado de Cortes de Caja')]
class Concentrado extends Component
{
    public $fechaInicio;
    public $fechaFin;
    public $idSucursal;
    public $idUsuario;
    public $opcionTurnos;
    
    public $idEmpresa;
    public $datosConcentrado = [];
    public $totalesGenerales = [];

    public function mount()
    {
        $this->idEmpresa = Auth::user()->tenant_id;
        $this->fechaInicio = $this->fechaInicio ?? date('Y-m-d');
        $this->fechaFin = $this->fechaFin ?? date('Y-m-d');
        $this->opcionTurnos = $this->opcionTurnos ?? 2;
        
        $this->cargarDatosConcentrado();
    }

    public function cargarDatosConcentrado()
    {
        // Obtener totales por sucursal
        $query = DB::table('cortes_turnos as c')
            ->select(
                's.razon_social as sucursal',
                DB::raw('COUNT(DISTINCT c.idturno) as total_turnos'),
                DB::raw('SUM(c.total_corte) as total_cortes'),
                DB::raw('COALESCE(SUM(ing.ingresos), 0) as total_ingresos'),
                DB::raw('COALESCE(SUM(eng.egresos), 0) as total_egresos'),
                DB::raw('COUNT(CASE WHEN c.turno_cerrado = 1 THEN 1 END) as turnos_cerrados'),
                DB::raw('COUNT(CASE WHEN c.turno_cerrado = 0 THEN 1 END) as turnos_abiertos')
            )
            ->leftJoin('cat_sucursales as s', 's.idsucursal', '=', 'c.idsucursal')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as ingresos
                FROM cortes_turnos_detalle
                WHERE tipo = 2
                GROUP BY idturno
            ) as ing"), 'ing.idturno', '=', 'c.idturn')
            ->leftJoin(DB::raw("(
                SELECT idturno, SUM(importe) as egresos
                FROM cortes_turnos_detalle
                WHERE tipo = 3
                GROUP BY idturno
            ) as eng"), 'eng.idturno', '=', 'c.idturn');

        // Aplicar filtros
        if ($this->fechaInicio) {
            $query->where('c.fechahora_apertura', '>=', $this->fechaInicio . ' 00:00:00');
        }

        if ($this->fechaFin) {
            $query->where('c.fechahora_apertura', '<=', $this->fechaFin . ' 23:59:59');
        }

        if ($this->idSucursal) {
            $query->where('c.idsucursal', $this->idSucursal);
        }

        if ($this->idUsuario) {
            $query->where('c.idusuario', $this->idUsuario);
        }

        if ($this->opcionTurnos == 0) {
            $query->where('c.turno_cerrado', 0);
        } elseif ($this->opcionTurnos == 1) {
            $query->where('c.turno_cerrado', 1);
        }

        $datosConcentrado = $query->groupBy('s.razon_social')->get();
        
        // Calcular totales generales
        $this->totalesGenerales = [
            'total_turnos' => $datosConcentrado->sum('total_turnos'),
            'total_cortes' => $datosConcentrado->sum('total_cortes'),
            'total_ingresos' => $datosConcentrado->sum('total_ingresos'),
            'total_egresos' => $datosConcentrado->sum('total_egresos'),
            'turnos_cerrados' => $datosConcentrado->sum('turnos_cerrados'),
            'turnos_abiertos' => $datosConcentrado->sum('turnos_abiertos'),
        ];
        
        $this->totalesGenerales['diferencia'] = 
            $this->totalesGenerales['total_ingresos'] - 
            $this->totalesGenerales['total_egresos'] - 
            $this->totalesGenerales['total_cortes'];
            
        // Convertir a array para Livewire
        $this->datosConcentrado = $datosConcentrado->toArray();
    }

    public function volver()
    {
        return redirect()->route('d-reportes-cortes-caja', [
            'fechaInicio' => $this->fechaInicio,
            'fechaFin' => $this->fechaFin,
            'idSucursal' => $this->idSucursal,
            'idUsuario' => $this->idUsuario,
            'opcionTurnos' => $this->opcionTurnos
        ]);
    }

    public function render()
    {
        return view('livewire.reportes.cortes-caja.concentrado');
    }
}
