<?php

namespace App\Livewire\PuntoDeVenta\PuntoVentas;

use Illuminate\Support\Facades\Log;

class PuntoVentasPartidaHandler
{
    private const DEFAULT_IVA = '16';
    private const DEFAULT_RETENCION_IVA = '10.67';

    private static function bcround(string $number, int $precision = 2): string
    {
        $scale = max(0, $precision) + 1;
        $increment = '0.' . str_repeat('0', max(0, $precision)) . '5';
        $comp = bccomp($number, '0', $scale);
        $adjusted = $comp >= 0
            ? bcadd($number, $increment, $scale)
            : bcsub($number, $increment, $scale);
        return bcadd($adjusted, '0', max(0, $precision));
    }

    private static function bcclamp(string $value, string $min, string $max, int $scale = 8): string
    {
        if (bccomp($value, $min, $scale) < 0) {
            return $min;
        }
        if (bccomp($value, $max, $scale) > 0) {
            return $max;
        }
        return $value;
    }
    public static function inicializarPartidaVacia(): array
    {
        return [
            'cantidad' => 1,
            'precio_unitario' => 0.0,
            'importe' => 0.0,
            'descuento1_porcentaje' => 0.0,
            'descuento1_importe' => 0.0,
            'descuento2_porcentaje' => 0.0,
            'descuento2_importe' => 0.0,
            'descuento3_porcentaje' => 0.0,
            'descuento3_importe' => 0.0,
            'descuento_total' => 0.0,
            'subtotal' => 0.0,
            'iva_porcentaje' => (float) self::DEFAULT_IVA,
            'iva_importe' => 0.0,
            'ieps_porcentaje' => 0.0,
            'ieps_importe' => 0.0,
            'retencion_isr_porcentaje' => 0.0,
            'retencion_isr_importe' => 0.0,
            'retencion_iva_porcentaje' => (float) self::DEFAULT_RETENCION_IVA,
            'retencion_iva_importe' => 0.0,
            'modalidad_ieps' => 'exento',
            'importe_total' => 0.0,
            'descripcion_adicional' => '',
            'clave' => '',
            'descripcion' => '',
        ];
    }

    public static function prepararPartidaParaEdicion(array $partida): array
    {
        $get = static fn($arr, $k, $alt = null) => $arr[$k] ?? $alt;

        return [
            'clave' => $get($partida, 'clave', ''),
            'descripcion' => $get($partida, 'descripcion', ''),
            'cantidad' => (int) ($get($partida, 'cantidad', 1) ?? 1),

            'precio_unitario' => (float) self::bcround((string) ($get($partida, 'precio_unitario', '0') ?? '0'), 2),
            'importe' => (float) self::bcround((string) ($get($partida, 'importe', '0') ?? '0'), 2),

            'descuento1_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento1_porcentaje', '0') ?? '0'), 2),
            'descuento1_importe' => (float) self::bcround((string) ($get($partida, 'descuento1_importe', '0') ?? '0'), 2),
            'descuento2_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento2_porcentaje', '0') ?? '0'), 2),
            'descuento2_importe' => (float) self::bcround((string) ($get($partida, 'descuento2_importe', '0') ?? '0'), 2),
            'descuento3_porcentaje' => (float) self::bcround((string) ($get($partida, 'descuento3_porcentaje', '0') ?? '0'), 2),
            'descuento3_importe' => (float) self::bcround((string) ($get($partida, 'descuento3_importe', '0') ?? '0'), 2),
            'descuento_total' => (float) self::bcround((string) ($get($partida, 'descuento_total', $get($partida, 'descuento', '0')) ?? '0'), 2),

            'subtotal' => (float) self::bcround((string) ($get($partida, 'subtotal', '0') ?? '0'), 2),

            'iva_porcentaje' => (float) self::bcround((string) ($get($partida, 'iva_porcentaje', self::DEFAULT_IVA) ?? self::DEFAULT_IVA), 2),
            'iva_importe' => (float) self::bcround((string) ($get($partida, 'iva_importe', $get($partida, 'iva_trasladado', '0')) ?? '0'), 2),
            'ieps_porcentaje' => (float) self::bcround((string) ($get($partida, 'ieps_porcentaje', '0') ?? '0'), 2),
            'ieps_importe' => (float) self::bcround((string) ($get($partida, 'ieps_importe', $get($partida, 'ieps_trasladado', '0')) ?? '0'), 2),
            'retencion_isr_porcentaje' => (float) self::bcround((string) ($get($partida, 'retencion_isr_porcentaje', $get($partida, 'isr_retenido_porcentaje', '0')) ?? '0'), 2),
            'retencion_isr_importe' => (float) self::bcround((string) ($get($partida, 'retencion_isr_importe', $get($partida, 'isr_retenido', '0')) ?? '0'), 2),
            'retencion_iva_porcentaje' => (float) self::bcround((string) ($get($partida, 'retencion_iva_porcentaje', self::DEFAULT_RETENCION_IVA) ?? self::DEFAULT_RETENCION_IVA), 2),
            'retencion_iva_importe' => (float) self::bcround((string) ($get($partida, 'retencion_iva_importe', $get($partida, 'iva_retenido', '0')) ?? '0'), 2),
            'modalidad_ieps' => $get($partida, 'modalidad_ieps', 'exento'),

            'importe_total' => (float) self::bcround((string) ($get($partida, 'importe_total', $get($partida, 'total', '0')) ?? '0'), 2),
            'descripcion_adicional' => $get($partida, 'descripcion_adicional', ''),
        ];
    }

    public static function recalcularPartida(array $partidaEditando): array
    {
        if (!$partidaEditando) {
            return self::inicializarPartidaVacia();
        }

        $precision = 8;

        $cantidad = (string) max(1, (int) ($partidaEditando['cantidad'] ?? 1));
        $precioUnitario = (string) max(0, (float) ($partidaEditando['precio_unitario'] ?? 0));

        $desc1 = self::bcclamp((string) ($partidaEditando['descuento1_porcentaje'] ?? '0'), '0', '100');
        $desc2 = self::bcclamp((string) ($partidaEditando['descuento2_porcentaje'] ?? '0'), '0', '100');
        $desc3 = self::bcclamp((string) ($partidaEditando['descuento3_porcentaje'] ?? '0'), '0', '100');
    $iva = self::bcclamp((string) ($partidaEditando['iva_porcentaje'] ?? self::DEFAULT_IVA), '0', '100');
        $ieps = (string) max(0, (float) ($partidaEditando['ieps_porcentaje'] ?? 0));
        $ret_isr = self::bcclamp((string) ($partidaEditando['retencion_isr_porcentaje'] ?? '0'), '0', '100');
    $ret_iva = self::bcclamp((string) ($partidaEditando['retencion_iva_porcentaje'] ?? self::DEFAULT_RETENCION_IVA), '0', '100');

        $importe = bcmul($cantidad, $precioUnitario, $precision);

        $base1 = $importe;
        $d1 = bcmul($base1, bcdiv($desc1, '100', $precision), $precision);
        $base2 = bcsub($base1, $d1, $precision);
        $d2 = bcmul($base2, bcdiv($desc2, '100', $precision), $precision);
        $base3 = bcsub($base2, $d2, $precision);
        $d3 = bcmul($base3, bcdiv($desc3, '100', $precision), $precision);

        $descTotal = bcadd(bcadd($d1, $d2, $precision), $d3, $precision);
        $subtotal = bcsub($importe, $descTotal, $precision);
        if (bccomp($subtotal, '0', $precision) < 0) { $subtotal = '0'; }

        $iva_imp = bcmul($subtotal, bcdiv($iva, '100', $precision), $precision);
        $ieps_imp = bcmul($subtotal, bcdiv($ieps, '100', $precision), $precision);
        $ret_isr_imp = bcmul($subtotal, bcdiv($ret_isr, '100', $precision), $precision);
        $ret_iva_imp = bcmul($subtotal, bcdiv($ret_iva, '100', $precision), $precision);

        $total = bcsub(
            bcadd(bcadd($subtotal, $iva_imp, $precision), $ieps_imp, $precision),
            bcadd($ret_isr_imp, $ret_iva_imp, $precision),
            $precision
        );

        $partidaEditando['cantidad'] = (int) $cantidad;
        $partidaEditando['precio_unitario'] = (float) self::bcround($precioUnitario, 2);
        $partidaEditando['descuento1_porcentaje'] = (float) self::bcround($desc1, 2);
        $partidaEditando['descuento2_porcentaje'] = (float) self::bcround($desc2, 2);
        $partidaEditando['descuento3_porcentaje'] = (float) self::bcround($desc3, 2);
        $partidaEditando['iva_porcentaje'] = (float) self::bcround($iva, 2);
        $partidaEditando['ieps_porcentaje'] = (float) self::bcround($ieps, 2);
        $partidaEditando['retencion_isr_porcentaje'] = (float) self::bcround($ret_isr, 2);
        $partidaEditando['retencion_iva_porcentaje'] = (float) self::bcround($ret_iva, 2);

        $partidaEditando['importe'] = (float) self::bcround($importe, 2);
        $partidaEditando['descuento1_importe'] = (float) self::bcround($d1, 2);
        $partidaEditando['descuento2_importe'] = (float) self::bcround($d2, 2);
        $partidaEditando['descuento3_importe'] = (float) self::bcround($d3, 2);
        $partidaEditando['descuento_total'] = (float) self::bcround($descTotal, 2);
        $partidaEditando['subtotal'] = (float) self::bcround($subtotal, 2);
        $partidaEditando['iva_importe'] = (float) self::bcround($iva_imp, 2);
        $partidaEditando['ieps_importe'] = (float) self::bcround($ieps_imp, 2);
        $partidaEditando['retencion_isr_importe'] = (float) self::bcround($ret_isr_imp, 2);
        $partidaEditando['retencion_iva_importe'] = (float) self::bcround($ret_iva_imp, 2);
        $partidaEditando['importe_total'] = (float) self::bcround($total, 2);

        return $partidaEditando;
    }

    public static function actualizarPartidaEnProductos(array &$productosagregados, int $indice, array $partidaEditando): bool
    {
        if (!isset($productosagregados[$indice])) {
            return false;
        }

        $partida = &$productosagregados[$indice];
        
        $calculosUnificados = PuntoVentasBaseLogic::sincronizarCalculosModal($partidaEditando);
        
        $partida['cantidad'] = $partidaEditando['cantidad'];
        $partida['precio_unitario'] = $partidaEditando['precio_unitario'];
        $partida['importe'] = $calculosUnificados['importe'];
        $partida['descuento'] = $partidaEditando['descuento_total'];
        $partida['subtotal'] = $calculosUnificados['subtotal'];
        $partida['iva_trasladado'] = $calculosUnificados['iva_trasladado'];
        $partida['ieps_trasladado'] = $calculosUnificados['ieps_trasladado'];
        $partida['iva_retenido'] = $calculosUnificados['iva_retenido'];
        $partida['isr_retenido'] = $calculosUnificados['isr_retenido'];
        $partida['total'] = $calculosUnificados['total'];
        $partida['descripcion_adicional'] = $partidaEditando['descripcion_adicional'];
        $partida['iva_porcentaje'] = $calculosUnificados['iva_porcentaje'];
        $partida['iva_retenido_porcentaje'] = $calculosUnificados['iva_retenido_porcentaje'];
        $partida['ieps_porcentaje'] = $calculosUnificados['ieps_porcentaje'];
        $partida['isr_retenido_porcentaje'] = $calculosUnificados['isr_retenido_porcentaje'];
        $partida['descuento1_porcentaje'] = $partidaEditando['descuento1_porcentaje'];
        $partida['descuento1_importe'] = $partidaEditando['descuento1_importe'];
        $partida['descuento2_porcentaje'] = $partidaEditando['descuento2_porcentaje'];
        $partida['descuento2_importe'] = $partidaEditando['descuento2_importe'];
        $partida['descuento3_porcentaje'] = $partidaEditando['descuento3_porcentaje'];
        $partida['descuento3_importe'] = $partidaEditando['descuento3_importe'];
        $partida['descuento_total'] = $partidaEditando['descuento_total'];
        $partida['iva_importe'] = $partidaEditando['iva_importe'];
        $partida['ieps_importe'] = $partidaEditando['ieps_importe'];
        $partida['retencion_isr_porcentaje'] = $partidaEditando['retencion_isr_porcentaje'];
        $partida['retencion_isr_importe'] = $partidaEditando['retencion_isr_importe'];
        $partida['retencion_iva_porcentaje'] = $partidaEditando['retencion_iva_porcentaje'];
        $partida['retencion_iva_importe'] = $partidaEditando['retencion_iva_importe'];
        $partida['modalidad_ieps'] = $partidaEditando['modalidad_ieps'];
        $partida['importe_total'] = $partidaEditando['importe_total'];
        
        return true;
    }
}
