<?php

namespace App\Livewire\PuntoDeVenta\PuntoVentas;

class PuntoVentasBaseLogic
{
    private const DEFAULT_IVA = '16';
    private const DEFAULT_RETENCION_IVA = '10.67';
    /**
     * Redondeo con BCMath al número de decimales indicado.
     */
    private static function bcround(string $number, int $precision = 2): string
    {
        $scale = max(0, $precision) + 1;
        $increment = '0.' . str_repeat('0', max(0, $precision)) . '5';
        $comp = bccomp($number, '0', $scale);
        $adjusted = $comp >= 0
            ? bcadd($number, $increment, $scale)
            : bcsub($number, $increment, $scale);
        return bcadd($adjusted, '0', max(0, $precision));
    }

    public static function calcularPrecioFinalSimple($precioAntesIva, $opciones = [])
    {
        $precision = 8;
        $precio = (string) $precioAntesIva;

    $ivaPorcentaje = (string) ($opciones['iva_porcentaje'] ?? self::DEFAULT_IVA);
    $ivaRetenidoPorcentaje = (string) ($opciones['iva_retenido_porcentaje'] ?? self::DEFAULT_RETENCION_IVA);
        $isrRetenidoPorcentaje = (string) ($opciones['isr_retenido_porcentaje'] ?? 0);

        $precioConIva = bcadd(
            $precio,
            bcmul($precio, bcdiv($ivaPorcentaje, '100', $precision), $precision),
            $precision
        );

        $retencionIva = bcmul($precio, bcdiv($ivaRetenidoPorcentaje, '100', $precision), $precision);
        $retencionIsr = bcmul($precio, bcdiv($isrRetenidoPorcentaje, '100', $precision), $precision);

        $precioFinal = bcsub(bcsub($precioConIva, $retencionIva, $precision), $retencionIsr, $precision);

        return (float) self::bcround($precioFinal, 2);
    }

    public static function calcularTotalSimple($cantidad, $precioFinal)
    {
        $precision = 8;
        $total = bcmul((string) $cantidad, (string) $precioFinal, $precision);
        return (float) self::bcround($total, 2);
    }

    public static function calcularTotalesSimplificados($productosAgregados)
    {
        $precision = 8;
        $totalFinal = '0.00';

        foreach ($productosAgregados as $producto) {
            $totalProducto = (string) ($producto['total'] ?? '0');
            $totalFinal = bcadd($totalFinal, $totalProducto, $precision);
        }

        return [
            'totalfinal' => (float) self::bcround($totalFinal, 2),
        ];
    }

    public static function formatearMoneda($valor, $decimales = 2)
    {
        if (is_numeric($valor)) {
            $formateado = rtrim(rtrim(number_format($valor, 8, '.', ','), '0'), '.');
            return $formateado === '' ? '0' : $formateado;
        }
        return number_format($valor, $decimales, '.', ',');
    }

    public static function sincronizarCalculosModal($partidaEditando)
    {
        $precision = 8;
        $cantidad = (string) max(1, (int) ($partidaEditando['cantidad'] ?? 1));
        $precioUnitario = (string) ($partidaEditando['precio_unitario'] ?? '0');
        $descuentoTotal = (string) ($partidaEditando['descuento_total'] ?? '0');
        $ivaImporte = (string) ($partidaEditando['iva_importe'] ?? '0');
        $iepsImporte = (string) ($partidaEditando['ieps_importe'] ?? '0');
        $retencionIvaImporte = (string) ($partidaEditando['retencion_iva_importe'] ?? '0');
        $retencionIsrImporte = (string) ($partidaEditando['retencion_isr_importe'] ?? '0');

        $importe = bcmul($cantidad, $precioUnitario, $precision);
        $subtotal = bcsub($importe, $descuentoTotal, $precision);
        $sumImpuestos = bcadd($ivaImporte, $iepsImporte, $precision);
        $sumRetenciones = bcadd($retencionIvaImporte, $retencionIsrImporte, $precision);
        $total = bcsub(bcadd($subtotal, $sumImpuestos, $precision), $sumRetenciones, $precision);

        return [
            'importe' => (float) self::bcround($importe, 2),
            'subtotal' => (float) self::bcround($subtotal, 2),
            'iva_trasladado' => (float) self::bcround($ivaImporte, 2),
            'ieps_trasladado' => (float) self::bcround($iepsImporte, 2),
            'iva_retenido' => (float) self::bcround($retencionIvaImporte, 2),
            'isr_retenido' => (float) self::bcround($retencionIsrImporte, 2),
            'total' => (float) self::bcround($total, 2),
            'iva_porcentaje' => (float) ($partidaEditando['iva_porcentaje'] ?? self::DEFAULT_IVA),
            'iva_retenido_porcentaje' => (float) ($partidaEditando['retencion_iva_porcentaje'] ?? self::DEFAULT_RETENCION_IVA),
            'ieps_porcentaje' => (float) ($partidaEditando['ieps_porcentaje'] ?? 0),
            'isr_retenido_porcentaje' => (float) ($partidaEditando['retencion_isr_porcentaje'] ?? 0),
        ];
    }
}
