<?php

namespace App\Livewire\PuntoDeVenta\PuntoVentas;

use App\Models\Cliente;
use App\Models\Sucursal;
use Illuminate\Support\Facades\Log;
use App\Traits\WithLivewireAlert;
use Livewire\Attributes\Title;
use Livewire\Component;
use Mpdf\Mpdf;
use App\Repositories\PuntoDeVenta\PuntoDeVentaRepository;

#[Title('Punto de Ventas')]
class Index extends Component
{
    use WithLivewireAlert;

    public $search = [
        'clientes' => '',
        'productos' => '',
    ];

    public $selectedCliente = null;
    public $selectedProducto = null;
    public $mostrarTablaClientes = false;
    public $mostrarTablaProductos = false;
    public $nombre = '';
    public $telefono = '';
    public $clave = '';
    public $rfc = '';
    public $fechaForm = '';
    public $colonia = '';
    public $sucursalid;
    public $serie = 'PV';
    public $fecha;
    public $sucursales = [];
    public $idclient = '';
    public $clientesearch = '';
    public $clientenombre = '';
    public $clienterfc = '';
    public $clientedireccion = '';
    public $productoid;
    public $productosearch = '';
    public $cantidad = 1;
    public $precio = 0;
    public $precioventaantesiva = 0;
    public $precioventaconiva = 0;
    public $descuento = 0;
    public $descripcion = '';
    public $descripcionadicional = '';
    public $productos = [];
    public $claveproductoservicio = '';
    public $descripcionclaveproductoservicio = '';
    public $claveunidadmedida = '';
    public $descripcionunidadmedida = '';
    public $ivatrasladado = 0;
    public $ivaretenido = 0;
    public $iepstrasladado = 0;
    public $isrretenido = 0;
    public $productosagregados = [];
    public $importetotal = 0;
    public $descuentototal = 0;
    public $subtotaltotal = 0;
    public $retivatotal = 0;
    public $retisrtotal = 0;
    public $iepstotal = 0;
    public $impuestototal = 0;
    public $totalfinal = 0;
    public $folio;

    public $mostrarModalPartida = false;
    public $indicePartidaEditando = null;
    public $partidaEditando = [];
    public $mostrarModalPago = false;
    public $formaPagoSeleccionada = null;
    public $formasPago = [
        ['id' => 1, 'nombre' => 'EFECTIVO'],
        ['id' => 2, 'nombre' => 'CHEQUE'],
        ['id' => 3, 'nombre' => 'TARJETA DEBITO'],
        ['id' => 4, 'nombre' => 'TARJETA CREDITO'],
        ['id' => 5, 'nombre' => 'CREDITO A 30 DIAS'],
    ];
    public $pagos = [];
    public $pagoActual = [
        'idtipopago' => 1,
        'tipopago' => 'EFECTIVO',
        'importe' => 0,
        'referencia' => null,
    ];

    public function mount()
    {

        $this->fecha = now()->format('Y-m-d');
        $this->fechaForm = now()->format('d-m-Y');
        $this->cargarSucursales();
        $this->sucursalid = null;
        $this->generarFolio();

    $this->setClientePublicoGeneral();

    }

    public function cargarSucursales()
    {
        $this->sucursales = Sucursal::orderBy('razon_social')
            ->pluck('razon_social', 'idsucursal')
            ->toArray();
    }

    public function generarFolio()
    {
        $fecha = now()->format('Ymd');
        $consecutivo = str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $this->folio = $this->serie . '-' . $fecha . '-' . $consecutivo;
    }

    private function obtenerClientes()
    {
    $query = \App\Models\Cliente::query();

        if (!empty($this->search['clientes'])) {
            $query->where(function ($q) {
                $q->where('nombre', 'like', '%' . $this->search['clientes'] . '%')
                    ->orWhere('rfc', 'like', '%' . $this->search['clientes'] . '%')
                    ->orWhere('telefono', 'like', '%' . $this->search['clientes'] . '%');
            });
        }

        return $query->orderBy('nombre')
            ->take(10)
            ->get();
    }

    private function obtenerProductos()
    {
    $query = \App\Models\Producto::query();

        if (!empty($this->search['productos'])) {
            $query->where(function ($q) {
                $q->where('clave', 'like', '%' . $this->search['productos'] . '%')
                    ->orWhere('descripcion', 'like', '%' . $this->search['productos'] . '%')
                    ->orWhere('codigodebarras', 'like', '%' . $this->search['productos'] . '%');
            });
        }

        $productos = $query->orderBy('clave')->take(10)->get();

        Log::info('Productos encontrados: ' . count($productos));

        return $productos;
    }

    private function setClientePublicoGeneral(): void
    {
        try {
            $cliente = \App\Models\Cliente::query()
                ->where('clave', 'PUBLICO EN GENERAL')
                ->orWhereIn('nombre', ['PÚBLICO EN GENERAL', 'PUBLICO EN GENERAL'])
                ->orderBy('nombre')
                ->first();

            if ($cliente) {
                $this->idclient = $cliente->idclient;
                $this->clientenombre = $cliente->nombre;
                $this->clienterfc = $cliente->rfc ?? '';
                $this->clientedireccion = $cliente->direccion_completa ?? $cliente->direccion ?? '';
                $this->search['clientes'] = $cliente->nombre;
                $this->mostrarTablaClientes = false;
                $this->selectedCliente = null;
            }
        } catch (\Throwable $e) {
            Log::warning('[PV] No se pudo seleccionar cliente público en general: ' . $e->getMessage());
        }
    }

    public function buscarClientes()
    {
        $this->mostrarTablaClientes = true;
        $this->mostrarTablaProductos = false;
        $this->selectedCliente = null;
    }

    public function buscarProductos()
    {
        $this->mostrarTablaProductos = true;
        $this->mostrarTablaClientes = false;
        $this->selectedProducto = null;
        $this->productos = $this->obtenerProductos();
    }

    public function updatedSearch($value, $key)
    {
        if ($key === 'clientes') {
            if (strlen(trim($value)) > 0) {
                $this->mostrarTablaClientes = true;
                $this->mostrarTablaProductos = false;
            } else {
                $this->mostrarTablaClientes = false;
            }
        }

        if ($key === 'productos') {
            if (strlen(trim($value)) > 0) {
                $this->mostrarTablaProductos = true;
                $this->mostrarTablaClientes = false;
                $this->productos = $this->obtenerProductos();
            } else {
                if (!$this->selectedProducto) {
                    $this->mostrarTablaProductos = false;
                }
            }
        }
    }

    public function abrirDropdownClientes()
    {
        $this->mostrarTablaClientes = true;
        $this->mostrarTablaProductos = false;
    }

    public function abrirDropdownProductos()
    {
        $this->mostrarTablaProductos = true;
        $this->mostrarTablaClientes = false;

        $productos = $this->obtenerProductos();
        $this->productos = $productos;
    }

    public function cerrarDropdownClientes()
    {
        $this->mostrarTablaClientes = false;
    }

    public function cerrarDropdownProductos()
    {
        $this->mostrarTablaProductos = false;
    }

    public function seleccionarCliente($clienteId)
    {
    $cliente = \App\Models\Cliente::where('idclient', $clienteId)->first();
        if ($cliente) {
            $this->idclient = $cliente->idclient;
            $this->clientenombre = $cliente->nombre;
            $this->clienterfc = $cliente->rfc ?? '';
            $this->clientedireccion = $cliente->direccion_completa ?? $cliente->direccion ?? '';
            $this->search['clientes'] = $cliente->nombre;
            $this->mostrarTablaClientes = false;
            $this->selectedCliente = null;
        }
    }

    public function seleccionarProducto($productoId)
    {
    $producto = \App\Models\Producto::where('idproduct', $productoId)->first();
        if ($producto) {
            $this->productoid = $producto->idproduct;
            $this->productosearch = $producto->clave . ' - ' . $producto->descripcion;
            $this->descripcion = $producto->descripcion;

            $this->precioventaantesiva = $producto->precioventaantesiva;
            $this->precioventaconiva = PuntoVentasBaseLogic::calcularPrecioFinalSimple($producto->precioventaantesiva);
            $this->precio = $this->precioventaconiva;

            $this->search['productos'] = $producto->clave . ' - ' . $producto->descripcion;
            $this->mostrarTablaProductos = true;
            $this->productos = $this->obtenerProductos();

            $this->claveproductoservicio = $producto->clave_prodserv ?? '';
            $this->descripcionclaveproductoservicio = $producto->prodserv ?? '';
            $this->claveunidadmedida = $producto->clave_unidadmedida ?? '';
            $this->descripcionunidadmedida = $producto->unidadmedida ?? '';

            $this->selectedProducto = [
                'idproduct' => $producto->idproduct,
                'clave' => $producto->clave,
                'descripcion' => $producto->descripcion,
            ];
        }
    }

    public function agregarProducto()
    {
        $this->validate([
            'productoid' => 'required',
            'cantidad' => 'required|numeric|min:1',
            'precio' => 'required|numeric|min:0',
        ]);

        $precioFinal = $this->precio;
        $total = PuntoVentasBaseLogic::calcularTotalSimple($this->cantidad, $precioFinal);

        $clave = '';
        if (!empty($this->search['productos']) && strpos($this->search['productos'], ' - ') !== false) {
            $clave = explode(' - ', $this->search['productos'])[0];
        } else {
            $producto = \App\Models\Producto::where('idproduct', $this->productoid)->first();
            $clave = $producto ? $producto->clave : '';
        }

        $nuevoProducto = [
            'producto_id' => $this->productoid,
            'clave' => $clave,
            'descripcion' => $this->descripcion,
            'cantidad' => $this->cantidad,
            'precio_unitario' => $precioFinal,
            'total' => $total,
            'clave_producto_servicio' => $this->claveproductoservicio,
            'descripcion_clave_producto_servicio' => $this->descripcionclaveproductoservicio,
            'clave_unidad_medida' => $this->claveunidadmedida,
            'descripcion_unidad_medida' => $this->descripcionunidadmedida,
            'descripcion_adicional' => $this->descripcionadicional,
        ];

        $this->productosagregados[] = $nuevoProducto;

        $this->calcularTotales();
        $this->limpiarFormularioProducto();
    }

    public function eliminarProducto($indice)
    {
        if (isset($this->productosagregados[$indice])) {
            unset($this->productosagregados[$indice]);
            $this->productosagregados = array_values($this->productosagregados);
            $this->calcularTotales();
        }
    }

    public function actualizarCantidad($indice)
    {
        if (isset($this->productosagregados[$indice])) {
            $nuevaCantidad = $this->productosagregados[$indice]['cantidad'];

            if ($nuevaCantidad > 0) {
                $precioUnitario = $this->productosagregados[$indice]['precio_unitario'];
                $this->productosagregados[$indice]['total'] = PuntoVentasBaseLogic::calcularTotalSimple(intval($nuevaCantidad), $precioUnitario);

                $this->calcularTotales();
            }
        }
    }

    public function updated($propertyName)
    {
        if (str_starts_with($propertyName, 'partidaEditando.')
            && isset($this->partidaEditando['cantidad'])
            && isset($this->partidaEditando['precio_unitario'])) {
            $this->partidaEditando['total'] = $this->partidaEditando['cantidad'] * $this->partidaEditando['precio_unitario'];
        }
        
        if ($propertyName === 'precio') {
            $this->precioventaconiva = $this->precio;
        }
        
        if ($propertyName === 'precioventaconiva') {
            $this->precio = $this->precioventaconiva;
        }
    }

    private function calcularTotales()
    {
        $totales = PuntoVentasBaseLogic::calcularTotalesSimplificados($this->productosagregados);
        $this->totalfinal = $totales['totalfinal'];
    }

    private function limpiarFormularioProducto()
    {
        $this->productoid = null;
        $this->selectedProducto = null;
        $this->cantidad = 1;
        $this->precio = 0;
        $this->precioventaantesiva = 0;
        $this->precioventaconiva = 0;
        $this->descuento = 0;
        $this->descripcion = '';
        $this->descripcionadicional = '';
        $this->claveproductoservicio = '';
        $this->descripcionclaveproductoservicio = '';
        $this->claveunidadmedida = '';
        $this->descripcionunidadmedida = '';
        $this->search['productos'] = '';
        $this->mostrarTablaClientes = false;
        $this->mostrarTablaProductos = false;
    }

    public function agregarPagoLinea()
    {
        $this->pagos[] = [
            'idtipopago' => 1,
            'tipopago' => collect($this->formasPago)->firstWhere('id', 1)['nombre'] ?? 'EFECTIVO',
            'importe' => 0,
            'referencia' => null,
        ];
    }

    public function eliminarPagoLinea($index)
    {
    if (isset($this->pagos[$index]) && count($this->pagos) > 1) {
            unset($this->pagos[$index]);
            $this->pagos = array_values($this->pagos);
        }
    }

    public function updatedPagos($value, $key)
    {
        if (str_ends_with($key, '.idtipopago')) {
            $parts = explode('.', $key);
            $idx = intval($parts[1]);
            $id = intval($value);
            $nombre = collect($this->formasPago)->firstWhere('id', $id)['nombre'] ?? null;
            if ($nombre !== null) {
                $this->pagos[$idx]['tipopago'] = $nombre;
            }
        }
        if (str_ends_with($key, '.importe')) {
            $parts = explode('.', $key);
            $idx = intval($parts[1]);
            $this->pagos[$idx]['importe'] = $this->normalizeMoney($value);
        }
    }

    public function updatedPagoActual($value, $key)
    {
        if ($key === 'idtipopago') {
            $id = intval($value);
            $nombre = collect($this->formasPago)->firstWhere('id', $id)['nombre'] ?? null;
            if ($nombre !== null) {
                $this->pagoActual['tipopago'] = $nombre;
            }
        }
        if ($key === 'importe') {
            $this->pagoActual['importe'] = $this->normalizeMoney($value);
        }
    }

    private function initPagoActualDefault(): void
    {
        $id = 1;
        $nombre = collect($this->formasPago)->firstWhere('id', $id)['nombre'] ?? 'EFECTIVO';
        $saldo = max(0, round(floatval($this->totalfinal) - $this->sumaPagos(), 2));
        $this->pagoActual = [
            'idtipopago' => $id,
            'tipopago' => $nombre,
            'importe' => $this->normalizeMoney($saldo),
            'referencia' => null,
        ];
    }

    public function agregarPagoDesdeActual()
    {
    $importe = round(floatval($this->pagoActual['importe'] ?? 0), 2);
        if ($importe <= 0) {
            session()->flash('error', 'El importe debe ser mayor a 0.');
            return;
        }

    $saldo = max(0, round(floatval($this->totalfinal) - $this->sumaPagos(), 2));
        if ($importe > $saldo + 0.01) {
            session()->flash('error', 'El importe no puede exceder el saldo pendiente.');
            return;
        }

        $id = intval($this->pagoActual['idtipopago'] ?? 1);
        $nombre = $this->pagoActual['tipopago'] ?? (collect($this->formasPago)->firstWhere('id', $id)['nombre'] ?? 'EFECTIVO');

        $this->pagos[] = [
            'idtipopago' => $id,
            'tipopago' => $nombre,
            'importe' => $this->normalizeMoney($importe),
            'referencia' => $this->pagoActual['referencia'] ?? null,
        ];

        $this->initPagoActualDefault();
    }

    public function saldoRestante(): float
    {
        $saldo = floatval($this->totalfinal) - $this->sumaPagos();
        return max(0.0, round($saldo, 2));
    }

    private function normalizeMoney($value): string
    {
        $v = round(floatval($value), 2);
        return number_format($v, 2, '.', '');
    }

    private function sumaPagos(): float
    {
        $sum = 0.0;
        foreach ($this->pagos as $p) {
            $sum += round(floatval($p['importe'] ?? 0), 2);
        }
        return round($sum, 2);
    }

    public function procesarVenta()
    {
        try {
            [$cliente, $sucursal] = $this->validarVentaPreconditions();

            if ($this->prepararPagosSiNecesario()) {
                return null;
            }
            
            if (! $this->validarPagosContraTotal()) {
                return null;
            }

            $payload = $this->buildVentaPayload();
            if (! $this->guardarVentaRepositorio($payload)) {
                return null;
            }

            $productosRepetidos = $this->prepararProductosParaImpresion();
            $datos = $this->construirDatosPdf($cliente, $sucursal, $productosRepetidos);
            $mpdf = $this->generarPdf($datos);

            session()->flash('message', 'Venta procesada exitosamente.');
            $this->limpiarTodo();
            $this->generarFolio();

            return response()->streamDownload(function () use ($mpdf) {
                $mpdf->Output('venta.pdf', 'D');
            }, 'venta_' . date('Ymd_His') . '.pdf');
        } catch (\Illuminate\Validation\ValidationException $e) {
            $errors = $e->validator->errors()->all();
            session()->flash('error', 'Error de validación: ' . implode(', ', $errors));
            Log::warning('Validación fallida en venta: ' . implode(', ', $errors));
            return null;
        } catch (\RuntimeException $e) {
            session()->flash('error', $e->getMessage());
            Log::info('Validación de negocio fallida: ' . $e->getMessage());
            return null;
        } catch (\Throwable $e) {
            Log::error('Error al procesar venta: ' . $e->getMessage());
            session()->flash('error', 'Error al procesar la venta. Por favor intente nuevamente.');
            return null;
        }
    }

    private function validarVentaPreconditions(): array
    {
   
        $this->validate([
            'idclient' => 'required|exists:clientes,idclient',
            'sucursalid' => 'required|exists:sucursals,id',
            'productosagregados' => 'required|array|min:1',
        ], [
            'idclient.required' => 'El cliente es obligatorio.',
            'idclient.exists' => 'El cliente seleccionado no es válido.',
            'sucursalid.required' => 'La sucursal es obligatoria.',
            'sucursalid.exists' => 'La sucursal seleccionada no es válida.',
            'productosagregados.required' => 'Debe agregar al menos un producto.',
            'productosagregados.min' => 'Debe agregar al menos un producto.',
        ]);


        
        $cliente = Cliente::where('idclient', $this->idclient)->first();
        
        $sucursal = Sucursal::find($this->sucursalid);

        
        return [$cliente, $sucursal];
    }

    private function prepararPagosSiNecesario(): bool
    {
        if (! $this->mostrarModalPago && count($this->pagos) === 0) {
            $this->initPagoActualDefault();
            $this->mostrarModalPago = true;
            return true;
        }
        return false;
    }

    private function validarPagosContraTotal(): bool
    {
        if (count($this->pagos) === 0) {
            session()->flash('error', 'Agregue al menos una forma de pago para continuar.');
            return false;
        }
        $suma = $this->sumaPagos();
        if (abs($suma - floatval($this->totalfinal)) > 0.01) {
            session()->flash('error', 'La suma de los pagos debe ser igual al total de la venta.');
            return false;
        }
        return true;
    }

    private function buildVentaPayload(): array
    {
        return [
            'idclient' => $this->idclient ?? null,
            'sucursalid' => $this->sucursalid ?? null,
            'folio' => $this->folio,
            'productosagregados' => $this->productosagregados,
            'totalentregado' => $this->totalfinal,
            'pagos' => $this->pagos,
        ];
    }

    private function guardarVentaRepositorio(array $payload): bool
    {
        try {
            $this->asegurarTenancy();
            $repo = new PuntoDeVentaRepository();
            $repoResult = $repo->crearVenta($payload);
            $usedFolio = $repoResult['folio'] ?? $this->folio;
            Log::info('[PV] Venta creada via repo, folio: ' . $usedFolio);
            $this->mostrarModalPago = false;
            $this->formaPagoSeleccionada = null;
            return true;
        } catch (\Throwable $e) {
            Log::error('[PV] Error guardando venta: ' . $e->getMessage());
            session()->flash('error', 'Error al guardar la venta: ' . $e->getMessage());
            return false;
        }
    }

    private function asegurarTenancy(): void
    {
        try {
            if (function_exists('tenancy') && (! tenancy()->initialized || tenant('id') !== 42)) {
                Log::info('[PV] Forzando tenancy en UI al tenant 42');
                tenancy()->initialize(42);
            }
        } catch (\Throwable $e) {
            Log::warning('[PV] No se pudo inicializar tenancy en UI: ' . $e->getMessage());
        }
    }

    private function prepararProductosParaImpresion(): array
    {
        return collect($this->productosagregados)->map(function ($producto) {
            return [
                'clave' => $producto['clave'] ?? $producto['producto_id'] ?? '',
                'cantidad' => $producto['cantidad'],
                'unidad' => $producto['clave_unidad_medida'] ?? 'PZA',
                'descripcion' => $producto['descripcion'],
                'precio_unitario' => $producto['precio_unitario'],
                'importe' => $producto['total'],
                'subtotal' => $producto['total'],
                'total' => $producto['total']
            ];
        })->toArray();
    }

    private function construirDatosPdf($cliente, $sucursal, array $productos): array
    {
        return [
            'folio' => $this->folio,
            'fecha' => now()->format('d/m/Y'),
            'empresa' => [
                'razon_social' => 'RAZON SOCIAL DE LA EMPRESA DEM',
                'rfc' => 'SAMJ770624411',
                'direccion' => 'CALLE 59 2448 X 118 Y 120 FRACC. YUCALPETEN',
                'ciudad' => 'MERIDA YUCATAN MEXICO',
                'cp' => '97248',
                'lugar_expedicion' => '97248 YUCATAN',
                'regimen_fiscal' => '621|REGIMEN DE INCORPORACION FISCAL',
                'telefono' => '01.999.945.99.99'
            ],
            'cliente' => [
                'sucursal' => $sucursal->razon_social,
                'nombre' => $cliente->nombre,
                'rfc' => $cliente->rfc ?? '',
                'direccion' => $cliente->direccion_completa ?? '',
                'cp' => $cliente->codigopostal ?? '',
                'regimen_fiscal' => '621|Incorporación Fiscal'
            ],
            'productos' => $productos,
            'totales' => [
                'subtotal' => $this->totalfinal,
                'iva' => 0,
                'ieps' => 0,
                'retencion_isr' => 0,
                'retencion_iva' => 0,
                'retenciones' => 0,
                'total' => $this->totalfinal
            ]
        ];
    }

    private function generarPdf(array $datos): Mpdf
    {
        $html = view('exports.punto-ventas', $datos)->render();
        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4',
            'orientation' => 'P'
        ]);
        $mpdf->WriteHTML($html);
        return $mpdf;
    }

    public function limpiarTodo()
    {
        $this->productosagregados = [];
        $this->limpiarFormularioProducto();
        $this->selectedCliente = null;
        $this->selectedProducto = null;
        $this->idclient = '';
        $this->clientenombre = '';
        $this->clienterfc = '';
        $this->clientedireccion = '';
        $this->search['clientes'] = '';
        $this->search['productos'] = '';
        $this->mostrarTablaClientes = false;
        $this->mostrarTablaProductos = false;
        $this->calcularTotales();
        $this->pagos = [];
        $this->mostrarModalPago = false;
        $this->formaPagoSeleccionada = null;
        try {
            $this->initPagoActualDefault();
        } catch (\Throwable $e) {
            Log::warning('[PV] Error al inicializar pago actual en limpiarTodo: ' . $e->getMessage());
        }

        $this->setClientePublicoGeneral();
    $this->generarFolio();
    }

    public function cancelarPago(): void
    {
        $this->mostrarModalPago = false;
        $this->pagos = [];
        try {
            $this->initPagoActualDefault();
        } catch (\Throwable $e) {
            Log::warning('[PV] Error al inicializar pago actual en cancelarPago: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.punto-de-venta.punto-ventas.index', [
            'clientes' => $this->obtenerClientes(),
            'productos' => $this->obtenerProductos(),
        ]);
    }
}
