<?php

namespace App\Livewire\Inventarios\Establecerstock;

use App\Models\Producto;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Closure;
use Illuminate\Support\Str;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;

#[Title('Establecer Stock')]
class Index extends Component
{
    use WithPagination;

    public int $refreshKey = 0;
    public ?string $fechaInicialVentas = null;
    public ?string $fechaFinalVentas = null;
    public ?int $marcaId = null;
    public ?int $familiaId = null;
    public ?int $sucursalId = null;
    public string $existenciaFiltro = 'Todos';
    public int $filas = 10;
    public bool $filtrarSinFechas = false;
    public bool $soloVendidos = false;
    public bool $incluirInactivos = false;
    public array $seleccion = [];
    public array $edits = [];
    public array $familias = [];
    public array $marcas = [];
    public array $sucursales = [];
    public array $existenciasOptions = [
        'Todos' => 'Todos',
        'ConExistencia' => 'Con existencia',
        'SinExistencia' => 'Sin existencia',
    ];

    public bool $mostrarModalFamilia = false;
    public bool $mostrarModalMarca = false;
    public ?int $familiaCambioId = null;
    public ?int $marcaCambioId = null;

    protected $paginationTheme = 'tailwind';
    private const NUMERIC_RULE = 'nullable|numeric|min:0';

    protected array $rules = [
        'edits.*.stock_minimo' => self::NUMERIC_RULE,
        'edits.*.costo_compra' => self::NUMERIC_RULE,
        'edits.*.precio_venta' => self::NUMERIC_RULE,
        'edits.*.activo' => 'boolean',
    ];

    public function mount(): void
    {
        $this->marcas = DB::table('cat_marcas')
            ->select('idbrand as id', 'marca as nombre')
            ->orderBy('marca')
            ->get()
            ->toArray();

        $this->familias = DB::table('cat_familias')
            ->select('idfamily as id', 'nombrefamilia as nombre')
            ->orderBy('nombrefamilia')
            ->get()
            ->toArray();

        $this->sucursales = DB::table('cat_sucursales')
            ->select('idsucursal as id', 'razon_social as nombre')
            ->orderBy('razon_social')
            ->get()
            ->toArray();

    $hoy = now()->toDateString();
    $this->fechaInicialVentas = $this->fechaInicialVentas ?: $hoy;
    $this->fechaFinalVentas = $this->fechaFinalVentas ?: $hoy;
    }

    protected function queryBase(): Builder
    {
    $almacenId = $this->sucursalId;

        $q = Producto::query()->select([
            'idproduct as id',
            'idfamilia',
            'idmarca',
            'descripcion',
            'clave',
            'codigodebarras as codigo_barras',
            'stockminimo as stock_minimo',
            'precioultimacompra as costo_compra',
            'precioventaconiva as precio_venta',
            'productoactivo as activo',
        ]);

        $this->addAggregatesSelects($q, $almacenId);
        $this->addLabelSelects($q);
        $this->applyExistenciaFilters($q);

            if ($this->marcaId !== null && $this->marcaId !== '') {
                $q->where('idmarca', (int) $this->marcaId);
            }
            if ($this->familiaId !== null && $this->familiaId !== '') {
                $q->where('idfamilia', (int) $this->familiaId);
            }

        return $q->orderBy('clave')->orderBy('idproduct');
    }

    private function addAggregatesSelects(Builder $q, $almacenId): void
    {
        $ex = DB::table('mov_inv_existencias as i')
            ->selectRaw('i.idproducto, SUM(i.existencia) as existencia');
        if ($almacenId) {
            $ex->whereIn('i.idalmacen', function ($sq) use ($almacenId) {
                $sq->from('cat_almacenes as a')
                    ->select('a.idwarehouse')
                    ->where('a.idsucursal', $almacenId);
            });
        }
        $ex->groupBy('i.idproducto');
        $q->leftJoinSub($ex, 'ex', 'ex.idproducto', '=', 'cat_productos.idproduct');
        $q->addSelect(DB::raw('COALESCE(ex.existencia,0) as existencia'));

        $v = DB::table('ing_ventasdetalle as vd')
            ->selectRaw('vd.idproducto, SUM(vd.cantidad) as ventas')
            ->where('vd.cantidad', '>', 0);
        if ($almacenId) {
            $v->where('vd.idsucursal', $almacenId);
        }
        if (! $this->filtrarSinFechas) {
            if ($this->fechaInicialVentas) {
                $inicio = $this->fechaInicialVentas . ' 00:00:00';
                $v->where('vd.fechahora', '>=', $inicio);
            }
            if ($this->fechaFinalVentas) {
                $fin = $this->fechaFinalVentas . ' 23:59:59';
                $v->where('vd.fechahora', '<=', $fin);
            }
        }
        $v->groupBy('vd.idproducto');
        $q->leftJoinSub($v, 'v', 'v.idproducto', '=', 'cat_productos.idproduct');
        $q->addSelect(DB::raw('COALESCE(v.ventas,0) as ventas'));
    }

    private function applyExistenciaFilters(Builder $q): void
    {
        if ($this->existenciaFiltro === 'ConExistencia') {
            $q->where('ex.existencia', '>', 0);
        }

        if ($this->existenciaFiltro === 'SinExistencia') {
            $q->where(function ($w) {
                $w->whereNull('ex.existencia')->orWhere('ex.existencia', '<=', 0);
            });
        }

        if (! $this->incluirInactivos) {
            $q->where('productoactivo', 1);
            $q->where('ex.existencia', '>', 0);
        }
        if ($this->soloVendidos) {
            $q->where('v.ventas', '>', 0);
        }
    }

    public function updated($name, $value): void
    {
        if ($this->handleFilterUpdated($name)) {
            return;
        }
        if ($this->handleBulkChangeUpdated($name, $value)) {
            return;
        }
        if ($name === 'filas') {
            $this->resetPage();
            return;
        }
        $this->markEditDirtyIfNeeded($name);
    }

    private function handleFilterUpdated(string $name): bool
    {
        $filtros = [
            'fechaInicialVentas', 'fechaFinalVentas', 'sucursalId', 'marcaId', 'familiaId',
            'existenciaFiltro', 'filtrarSinFechas', 'soloVendidos', 'incluirInactivos',
        ];
        if (!in_array($name, $filtros, true)) {
            return false;
        }
        if (in_array($name, ['sucursalId','marcaId','familiaId'], true)) {
            $this->{$name} = ($this->{$name} === '' || $this->{$name} === null) ? null : (int) $this->{$name};
        }
        $this->edits = [];
        $this->seleccion = [];
        $this->refreshKey++;
        $this->resetPage();
        try { Cache::forget($this->totalsCacheKey()); } catch (\Throwable $e) { /* noop */ }
        return true;
    }

    private function handleBulkChangeUpdated(string $name, $value): bool
    {
        if ($name === 'familiaCambioId') {
            $familia = ($value === '' || $value === null) ? null : (int) $value;
            if ($familia !== null) {
                foreach ($this->seleccion as $id) {
                    $id = (int) $id;
                    $this->edits[$id]['familia_aplicada'] = $familia;
                    $this->edits[$id]['_dirty'] = true;
                }
            }
            return true;
        }
        if ($name === 'marcaCambioId') {
            $marca = ($value === '' || $value === null) ? null : (int) $value;
            if ($marca !== null) {
                foreach ($this->seleccion as $id) {
                    $id = (int) $id;
                    $this->edits[$id]['marca_aplicada'] = $marca;
                    $this->edits[$id]['_dirty'] = true;
                }
            }
            return true;
        }
        return false;
    }

    private function markEditDirtyIfNeeded($name): void
    {
        if (!Str::startsWith((string) $name, 'edits.')) {
            return;
        }
        $parts = explode('.', (string) $name);
        if (count($parts) >= 3) {
            $id = (int) $parts[1];
            $this->edits[$id]['_dirty'] = true;
        }
    }

    public function filtroChanged(): void
    {
        $this->edits = [];
        $this->seleccion = [];
        $this->refreshKey++;
        $this->resetPage();
        try { Cache::forget($this->totalsCacheKey()); } catch (\Throwable $e) { /* noop */ }
    }

    public function guardarCambios(): void
    {
        $this->validate();
        $ids = array_keys($this->edits);
        if (empty($ids)) {
            session()->flash('error', 'No hay cambios que guardar.');

            return;
        }

        $productos = Producto::whereIn('idproduct', $ids)->get();
        $totalActualizados = 0;
        foreach ($productos as $prod) {
            $totalActualizados += $this->aplicarCambiosProducto($prod);
        }

        session()->flash('message', "$totalActualizados producto(s) actualizados.");
        foreach ($this->edits as &$row) {
            unset($row['_dirty']);
        }
    try { Cache::forget($this->totalsCacheKey()); } catch (\Throwable $e) { /* noop */ }
    }

    private function aplicarCambiosProducto(Producto $prod): int
    {
        $idVista = $prod->idproduct;
        $data = $this->edits[$idVista] ?? [];
        if (! Arr::get($data, '_dirty')) {
            return 0;
        }
        $map = [
            'clave' => 'clave',
            'codigo_barras' => 'codigodebarras',
            'stock_minimo' => 'stockminimo',
            'costo_compra' => 'precioultimacompra',
            'precio_venta' => 'precioventaconiva',
            'activo' => 'productoactivo',
            'familia_aplicada' => 'idfamilia',
            'marca_aplicada' => 'idmarca',
        ];
        $update = [];
        foreach ($map as $k => $col) {
            if (array_key_exists($k, $data)) {
                if ($k === 'activo') {
                    $update[$col] = $data[$k] ? 1 : 0;
                } else {
                    $update[$col] = $data[$k];
                }
            }
        }
        if (empty($update)) {
            return 0;
        }
        $prod->fill($update)->save();

        return 1;
    }

    public function toggleSeleccionTodos(): void
    {
        $idsPagina = collect($this->queryBase()->Paginate($this->filas)->items())
            ->pluck('id')->map(fn ($v) => (int) $v)->all();

        $setSeleccion = collect($this->seleccion)->map(fn ($v) => (int) $v)->flip();
        $todosSeleccionados = collect($idsPagina)->every(fn ($id) => $setSeleccion->has($id));

        if ($todosSeleccionados) {
            $this->seleccion = collect($this->seleccion)
                ->map(fn ($v) => (int) $v)
                ->reject(fn ($id) => in_array($id, $idsPagina, true))
                ->values()
                ->all();
        } else {
            $this->seleccion = collect($this->seleccion)
                ->map(fn ($v) => (int) $v)
                ->merge($idsPagina)
                ->unique()
                ->values()
                ->all();
        }
    }

    public function toggleActivoSeleccionados(): void
    {
        foreach ($this->seleccion as $id) {
            $id = (int) $id;
            if (! isset($this->edits[$id])) {
                $p = Producto::select('idproduct','clave','descripcion','codigodebarras','stockminimo','precioultimacompra','precioventaconiva','productoactivo')
                    ->where('idproduct', $id)->first();
                if (! $p) { continue; }
                $this->edits[$id] = [
                    'clave' => $p->clave,
                    'descripcion' => $p->descripcion,
                    'codigo_barras' => $p->codigodebarras,
                    'stock_minimo' => $p->stockminimo,
                    'costo_compra' => $p->precioultimacompra,
                    'precio_venta' => $p->precioventaconiva,
                    'existencia' => 0,
                    'activo' => (bool) $p->productoactivo,
                ];
            }
            $this->edits[$id]['activo'] = ! (bool) ($this->edits[$id]['activo'] ?? false);
            $this->edits[$id]['_dirty'] = true;
        }
    }

    public function cambiarFamiliaSeleccionados(): void
    {
        $this->familiaCambioId = $this->familiaId;
        $this->mostrarModalFamilia = true;
    }

    public function cambiarMarcaSeleccionados(): void
    {
        $this->marcaCambioId = $this->marcaId;
        $this->mostrarModalMarca = true;
    }

    public function restaurarTodos(): void
    {
        $ids = collect($this->seleccion)->map(fn ($v) => (int) $v)->values()->all();
        if (empty($ids)) {
            session()->flash('error', 'No hay productos seleccionados para restaurar.');
            return;
        }
        $productos = Producto::whereIn('idproduct', $ids)->get(['idproduct','clave','descripcion','codigodebarras','stockminimo','precioultimacompra','precioventaconiva','productoactivo','idfamilia','idmarca']);
        foreach ($productos as $p) {
            $pid = (int) $p->idproduct;
            $this->edits[$pid] = [
                'clave' => $p->clave,
                'descripcion' => $p->descripcion,
                'codigo_barras' => $p->codigodebarras,
                'stock_minimo' => $p->stockminimo,
                'costo_compra' => $p->precioultimacompra,
                'precio_venta' => $p->precioventaconiva,
                'existencia' => $this->edits[$pid]['existencia'] ?? 0,
                'familia_aplicada' => $p->idfamilia,
                'marca_aplicada' => $p->idmarca,
                'activo' => (bool) $p->productoactivo,
            ];
        }
    }

    public function confirmarCambiarFamilia(): void
    {
        if ($this->familiaCambioId === null || $this->familiaCambioId === '') {
            session()->flash('error', 'Selecciona una familia destino.');
            return;
        }
        if (empty($this->seleccion)) {
            session()->flash('error', 'Selecciona al menos un producto.');
            return;
        }
        $familia = (int) $this->familiaCambioId;
        $aplicados = 0;
        foreach ($this->seleccion as $id) {
            $id = (int) $id;
            $this->edits[$id]['familia_aplicada'] = $familia;
            $this->edits[$id]['_dirty'] = true;
            $aplicados++;
        }
        $this->mostrarModalFamilia = false;
        session()->flash('message', "$aplicados producto(s) marcados para cambiar de familia.");
    }

    public function confirmarCambiarMarca(): void
    {
        if ($this->marcaCambioId === null || $this->marcaCambioId === '') {
            session()->flash('error', 'Selecciona una marca destino.');
            return;
        }
        if (empty($this->seleccion)) {
            session()->flash('error', 'Selecciona al menos un producto.');
            return;
        }
        $marca = (int) $this->marcaCambioId;
        $aplicados = 0;
        foreach ($this->seleccion as $id) {
            $id = (int) $id;
            $this->edits[$id]['marca_aplicada'] = $marca;
            $this->edits[$id]['_dirty'] = true;
            $aplicados++;
        }
        $this->mostrarModalMarca = false;
        session()->flash('message', "$aplicados producto(s) marcados para cambiar de marca.");
    }

    public function cerrarModales(): void
    {
        $this->mostrarModalFamilia = false;
        $this->mostrarModalMarca = false;
    }

    

    public function render()
    {
        $baseQuery = $this->queryBase();
        $productos = (clone $baseQuery)->Paginate($this->filas);
        foreach ($productos as $p) {
            if (! isset($this->edits[$p->id])) {
                $this->edits[$p->id] = [
                    'clave' => $p->clave,
                    'descripcion' => $p->descripcion,
                    'codigo_barras' => $p->codigo_barras,
                    'stock_minimo' => $p->stock_minimo,
                    'costo_compra' => $p->costo_compra,
                    'precio_venta' => $p->precio_venta,
                    'existencia' => $p->existencia ?? 0,
                    'familia_aplicada' => $p->idfamilia ?? null,
                    'marca_aplicada' => $p->idmarca ?? null,
                    'activo' => (bool) $p->activo,
                ];
            }
        }

        $sub = $baseQuery->toBase();
        $agg = $this->rememberTotals($this->totalsCacheKey(), 60, function () use ($sub) {
            return DB::query()->fromSub($sub, 't')
            ->selectRaw(implode(', ', [
                'COALESCE(SUM(t.stock_minimo * t.precio_venta),0) as venta_stockmin_total',
                'COALESCE(SUM(t.existencia * t.precio_venta),0) as venta_existencia_total',
                'COALESCE(SUM(COALESCE(t.ventas,0) * t.precio_venta),0) as venta_ventas_total',
                'COALESCE(SUM(t.stock_minimo * t.costo_compra),0) as compra_stockmin_total',
                'COALESCE(SUM(t.existencia * t.costo_compra),0) as compra_existencia_total',
                'COALESCE(SUM(COALESCE(t.ventas,0) * t.costo_compra),0) as compra_ventas_total',
                'COALESCE(SUM(t.stock_minimo),0) as unidades_stockmin',
                'COALESCE(SUM(t.existencia),0) as unidades_existencia',
                'COALESCE(SUM(COALESCE(t.ventas,0)),0) as unidades_ventas',
            ]))
            ->first();
        });

        $totals = [
            'venta' => [
                'stockmin_total' => (float) ($agg->venta_stockmin_total ?? 0),
                'existencia_total' => (float) ($agg->venta_existencia_total ?? 0),
                'ventas_total' => (float) ($agg->venta_ventas_total ?? 0),
            ],
            'compra' => [
                'stockmin_total' => (float) ($agg->compra_stockmin_total ?? 0),
                'existencia_total' => (float) ($agg->compra_existencia_total ?? 0),
                'ventas_total' => (float) ($agg->compra_ventas_total ?? 0),
            ],
            'units' => [
                'stockmin' => (float) ($agg->unidades_stockmin ?? 0),
                'existencia' => (float) ($agg->unidades_existencia ?? 0),
                'ventas' => (float) ($agg->unidades_ventas ?? 0),
            ],
            'perdida_compra' => (
                (float) ($agg->compra_existencia_total ?? 0)
                + (float) ($agg->compra_ventas_total ?? 0)
                - (float) ($agg->compra_stockmin_total ?? 0)
            ),
        ];

        return view('livewire.Inventarios.establecerstock.index', [
            'productos' => $productos,
            'totals' => $totals,
        ]);

    }

    private function totalsCacheKey(): string
    {
        $parts = [
            'suc' => $this->sucursalId,
            'fi' => $this->fechaInicialVentas,
            'ff' => $this->fechaFinalVentas,
            'sinF' => $this->filtrarSinFechas ? 1 : 0,
            'vend' => $this->soloVendidos ? 1 : 0,
            'inact' => $this->incluirInactivos ? 1 : 0,
            'marca' => $this->marcaId,
            'fam' => $this->familiaId,
            'exist' => $this->existenciaFiltro,
        ];
        return 'eststock:totals:' . md5(json_encode($parts));
    }

    private function rememberTotals(string $key, int $seconds, Closure $callback)
    {
        try {
            return Cache::remember($key, $seconds, $callback);
        } catch (\Throwable $e) {
            if (stripos($e->getMessage(), 'does not support tagging') !== false) {
                return $callback();
            }
            throw $e;
        }
    }

    private function addLabelSelects(Builder $q): void
    {
        $q->selectSub(function ($sub) {
            $sub->from('cat_familias as f')
                ->selectRaw("COALESCE(f.nombrefamilia, '')")
                ->whereColumn('f.idfamily', 'cat_productos.idfamilia')
                ->limit(1);
        }, 'familia_nombre');
        $q->selectSub(function ($sub) {
            $sub->from('cat_marcas as m')
                ->selectRaw("COALESCE(m.marca, '')")
                ->whereColumn('m.idbrand', 'cat_productos.idmarca')
                ->limit(1);
        }, 'marca_nombre');
    }

}
