<?php

namespace App\Livewire\Inventarios\CambiarPrecios;

use App\Models\Producto;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;

#[Title('Cambiar precios')]
class Index extends Component
{
    use WithPagination;

    public ?int $familiaId = null;
    public ?int $marcaId = null;
    public bool $incluirInactivos = false;
    public bool $mostrarProductos = true;
    public string $aplicarPor = 'porcentaje';
    public ?float $cantidad = null;
    public string $modoSeleccion = 'contemplados';
    public array $familias = [];
    public array $marcas = [];
    public array $seleccion = [];
    public int $filas = 25;

    public function mount(): void
    {
        $this->familias = DB::table('cat_familias')
            ->select('idfamily as id', 'nombrefamilia as nombre')
            ->orderBy('nombrefamilia')
            ->get()->toArray();

        $this->marcas = DB::table('cat_marcas')
            ->select('idbrand as id', 'marca as nombre')
            ->orderBy('marca')
            ->get()->toArray();
    }

    public function updating($name, $value)
    {
        if (in_array($name, ['familiaId','marcaId','incluirInactivos','filas'], true)) {
            $this->resetPage();
            $this->seleccion = [];
        }
    }

    private function baseQuery(): Builder
    {
        $q = Producto::query()
            ->select([
                'idproduct as id',
                'descripcion',
                'clave',
                'codigodebarras as codigo_barras',
                'idfamilia',
                'idmarca',
                'productoactivo as activo',
                'precioventaconiva as precio_venta',
            ]);

        if (!$this->incluirInactivos) {
            $q->where('productoactivo', 1);
        }
        if ($this->familiaId) {
            $q->where('idfamilia', (int) $this->familiaId);
        }
        if ($this->marcaId) {
            $q->where('idmarca', (int) $this->marcaId);
        }

        $q->selectSub(function ($sub) {
            $sub->from('cat_familias as f')
                ->selectRaw("COALESCE(f.nombrefamilia, 'SIN FAMILIA')")
                ->whereColumn('f.idfamily', 'cat_productos.idfamilia')
                ->limit(1);
        }, 'familia_nombre');
        $q->selectSub(function ($sub) {
            $sub->from('cat_marcas as m')
                ->selectRaw("COALESCE(m.marca, 'SINMARCA')")
                ->whereColumn('m.idbrand', 'cat_productos.idmarca')
                ->limit(1);
        }, 'marca_nombre');

        return $q->orderBy('descripcion')->orderBy('idproduct');
    }

    public function toggleMostrarOcultarProductos(): void
    {
        $this->mostrarProductos = !$this->mostrarProductos;
    }

    public function toggleSeleccionTodos(): void
    {
        $idsPagina = collect($this->baseQuery()->simplePaginate($this->filas)->items())
            ->pluck('id')->map(fn ($v) => (int) $v)->all();

        $conjunto = collect($this->seleccion)->map(fn ($v) => (int) $v)->flip();
        $todos = collect($idsPagina)->every(fn ($id) => $conjunto->has($id));

        if ($todos) {
            $this->seleccion = collect($this->seleccion)
                ->map(fn ($v) => (int) $v)
                ->reject(fn ($id) => in_array($id, $idsPagina, true))
                ->values()->all();
        } else {
            $this->seleccion = collect($this->seleccion)
                ->map(fn ($v) => (int) $v)
                ->merge($idsPagina)->unique()->values()->all();
        }
    }

    public function aplicarPrecios(): void
    {
        $this->validate([
            'aplicarPor' => 'required|in:porcentaje,monto',
            'cantidad' => 'required|numeric',
        ]);

        $query = $this->baseQuery();
        $idsSeleccionados = collect($this->seleccion)->map(fn ($v) => (int) $v)->all();

        if ($this->modoSeleccion === 'contemplados' && !empty($idsSeleccionados)) {
            $query->whereIn('idproduct', $idsSeleccionados);
        } elseif ($this->modoSeleccion === 'omitidos' && !empty($idsSeleccionados)) {
            $query->whereNotIn('idproduct', $idsSeleccionados);
        }

        $ids = $query->limit(100000)->pluck('id')->all();
        if (empty($ids)) {
            $this->dispatch('error-alert',
                title: '¡Atención!',
                text: 'No hay productos para actualizar con los criterios actuales.'
            );
            return;
        }

        $porcentaje = $this->aplicarPor === 'porcentaje';
        $monto = (float) $this->cantidad;

        $chunk = 1000;
        $total = 0;
        foreach (array_chunk($ids, $chunk) as $parte) {
            $productos = Producto::whereIn('idproduct', $parte)->get(['idproduct','precioventaconiva']);
            foreach ($productos as $p) {
                $precio = (float) $p->precioventaconiva;
                $nuevo = $porcentaje ? ($precio * (1 + $monto / 100)) : ($precio + $monto);
                if ($nuevo < 0) { $nuevo = 0; }
                if (abs($nuevo - $precio) < 0.0001) { continue; }
                $p->precioventaconiva = round($nuevo, 2);
                $p->save();
                $total++;
            }
        }

        $this->dispatch('success-alert',
            title: '¡Éxito!',
            text: "$total producto(s) actualizados correctamente"
        );
    }

    public function render()
    {
        $productos = $this->mostrarProductos
            ? $this->baseQuery()->Paginate($this->filas)
            : collect([]);

        return view('livewire.Inventarios.cambiar-precios.index', [
            'productos' => $productos,
            'totalEncontrados' => $this->baseQuery()->count(),
        ]);
    }
}
