<?php

namespace App\Livewire;

use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Livewire\Component;

class Dashboard extends Component
{
    public $totalventasxempresa = [];
    public $ventasxhora = [];
    public $totalventasxmes = [];
    public $totalventasxanio = [];
    public $totalventasglobal = [];

    // Fechas para emulación (cambiar a null para usar fecha actual)
    public $fechaEmulada = '2024-11-15';
    public $mesEmulado = 11;
    public $anioEmulado = 2024;

    public function mount()
    {
        $this->loadDashboardData();
    }

    public function loadDashboardData()
    {
        $this->loadVentasPorEmpresa();
        $this->loadVentasPorHora();
        $this->loadVentasPorDiaDelMes();
        $this->loadVentasPorMesDelAnio();
        $this->loadVentasPorAnio();
    }

    /**
     * Obtiene las ventas totales por sucursal del día actual
     * Combina datos de ventasgeneral e ing_ventasgeneral01
     */
    private function loadVentasPorEmpresa()
    {
        $today = $this->fechaEmulada ?? Carbon::today()->toDateString();

        // Subconsulta para ventasgeneral
        $ventasgeneral = DB::table('ventasgeneral as v')
            ->join('cat_sucursales as s', 's.idbranchoffice', '=', 'v.idsucursal')
            ->select(
                's.idbranchoffice',
                's.razon_social as sucursal',
                DB::raw('SUM(v.total) as total'),
                DB::raw('COUNT(v.folio) as totalregistros')
            )
            ->where('v.idstatus', '!=', 3)
            ->whereDate('v.fechaventa', $today)
            ->groupBy('s.idbranchoffice', 's.razon_social');

        // Subconsulta para ing_ventasgeneral01 (facturas)
        $facturas = DB::table('ing_ventasgeneral01 as v')
            ->join('cat_sucursales as s', 's.idbranchoffice', '=', 'v.idsucursal')
            ->select(
                's.idbranchoffice',
                's.razon_social as sucursal',
                DB::raw('SUM(v.totalventa) as total'),
                DB::raw('COUNT(v.numerodocumento) as totalregistros')
            )
            ->where('v.idstatus', '!=', 3)
            ->whereDate('v.fechaventa', $today)
            ->groupBy('s.idbranchoffice', 's.razon_social');

        // Combinar ambas consultas sumando los totales
        $this->totalventasxempresa = DB::table(DB::raw("({$ventasgeneral->toSql()}) as t1"))
            ->mergeBindings($ventasgeneral)
            ->join(DB::raw("({$facturas->toSql()}) as t2"), 't1.idbranchoffice', '=', 't2.idbranchoffice')
            ->mergeBindings($facturas)
            ->select(
                't1.idbranchoffice',
                't1.sucursal',
                DB::raw('(t1.total + t2.total) as total'),
                DB::raw('(t1.totalregistros + t2.totalregistros) as totalregistros')
            )
            ->orderBy('t1.sucursal')
            ->get();
    }

    /**
     * Obtiene las ventas del día agrupadas por hora
     * Combina ventasgeneral e ing_ventasgeneral01 usando UNION
     */
    private function loadVentasPorHora()
    {
        $today = $this->fechaEmulada ?? Carbon::today()->toDateString();

        // Usar raw query con UNION para combinar ambas tablas
        $this->ventasxhora = DB::select("
            (SELECT EXTRACT(HOUR FROM v.horaventa) as horaventa, SUM(v.total) as totalxhora
             FROM ventasgeneral v
             WHERE v.idstatus != 3 AND v.fechaventa = ?
             GROUP BY EXTRACT(HOUR FROM v.horaventa))
            UNION
            (SELECT EXTRACT(HOUR FROM v.fechahoradocumento) as horaventa, SUM(v.totalventa) as totalxhora
             FROM ing_ventasgeneral01 v
             WHERE v.idstatus != 3 AND v.fechaventa = ?
             GROUP BY EXTRACT(HOUR FROM v.fechahoradocumento))
            ORDER BY horaventa
        ", [$today, $today]);
    }

    /**
     * Obtiene las ventas del mes actual agrupadas por día
     * Combina ventasgeneral e ing_ventasgeneral01 usando UNION
     */
    private function loadVentasPorDiaDelMes()
    {
        $currentMonth = $this->mesEmulado ?? Carbon::now()->month;
        $currentYear = $this->anioEmulado ?? Carbon::now()->year;

        $this->totalventasxmes = DB::select("
            (SELECT EXTRACT(DAY FROM v.fechaventa) as diaventa, SUM(v.total) as totalxdia
             FROM ventasgeneral v
             WHERE v.idstatus != 3
               AND EXTRACT(YEAR FROM v.fechaventa) = ?
               AND EXTRACT(MONTH FROM v.fechaventa) = ?
             GROUP BY EXTRACT(DAY FROM v.fechaventa))
            UNION
            (SELECT EXTRACT(DAY FROM v.fechaventa) as diaventa, SUM(v.totalventa) as totalxdia
             FROM ing_ventasgeneral01 v
             WHERE v.idstatus != 3
               AND EXTRACT(YEAR FROM v.fechaventa) = ?
               AND EXTRACT(MONTH FROM v.fechaventa) = ?
             GROUP BY EXTRACT(DAY FROM v.fechaventa))
            ORDER BY diaventa
        ", [$currentYear, $currentMonth, $currentYear, $currentMonth]);
    }

    /**
     * Obtiene las ventas del año actual agrupadas por mes
     * Solo usa ventasgeneral (como en el código original)
     */
    private function loadVentasPorMesDelAnio()
    {
        $currentYear = $this->anioEmulado ?? Carbon::now()->year;

        $this->totalventasxanio = DB::select("
            SELECT EXTRACT(MONTH FROM v.fechaventa) as mesventa, SUM(v.total) as totalxmes
            FROM ventasgeneral v
            WHERE v.idstatus != 3 AND EXTRACT(YEAR FROM v.fechaventa) = ?
            GROUP BY EXTRACT(MONTH FROM v.fechaventa)
            ORDER BY mesventa
        ", [$currentYear]);
    }

    /**
     * Obtiene las ventas globales agrupadas por año
     * Solo usa ventasgeneral (como en el código original)
     */
    private function loadVentasPorAnio()
    {
        $this->totalventasglobal = DB::select("
            SELECT EXTRACT(YEAR FROM v.fechaventa) as anioventa, SUM(v.total) as totalxanio
            FROM ventasgeneral v
            WHERE v.idstatus != 3
            GROUP BY EXTRACT(YEAR FROM v.fechaventa)
            ORDER BY anioventa DESC
        ");
    }

    public function render()
    {
        return view('livewire.dashboard');
    }
}
