<?php

namespace App\Livewire\Catalogos\Productos;

use App\Enums\OrigenRegistro;
use App\Enums\Status;
use App\Models\Almacen;
use App\Models\BitacoraExistencia;
use App\Models\Concepto;
use App\Models\Familia;
use App\Models\Marca;
use App\Models\MovimientoInventarioDetalle;
use App\Models\MovimientoInventarioExistencia;
use App\Models\MovimientoInventarioGeneral;
use App\Models\Producto;
use App\Models\ProductoCompuesto;
use App\Models\ProductoServicio;
use App\Models\Unidad;
use App\Support\StringUtils;
use App\Traits\WithLivewireAlert;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Stringable;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;

#[Title('Editando producto')]
class Edit extends Component
{
    use ProductBaseLogic;
    use WithFileUploads;
    use WithPagination;
    use WithLivewireAlert;

    public $action = 'edit';
    public $families;
    public $brands;
    public $warehouses;
    public $productComponents = [];
    public $inventarios;
    public $originalImageURL;

    public $historicoEntrada = [
        'rango_fecha_activo' => false,
        'incluir_mov_inventarios' => false,
        'fecha_inicio' => null,
        'fecha_final' => null,
        'total' => 0,
        'sumatoria' => 0,
        'esta_buscando' => false
    ];

    public $historicoSalida = [
        'rango_fecha_activo' => false,
        'incluir_mov_inventarios' => false,
        'fecha_inicio' => null,
        'fecha_final' => null,
        'total' => 0,
        'sumatoria' => 0,
        'esta_buscando' => false
    ];

    public $idproduct;
    public function mount(Producto $producto)
    {
        $this->idproduct = $producto->idproduct;
        $this->form = $producto->toArray();
        $this->families = Familia::all();
        $this->brands = Marca::all();
        $this->warehouses = Almacen::with('sucursal')->get()->toArray();
        $this->codigo = $this->form['codigodebarras'];
        $this->productComponents = ProductoCompuesto::with(['producto:idproduct,clave,descripcion'])
            ->where('idproducto', $producto->idproduct)
            ->get()->toArray();


        if ($producto->manejainventario) {
            $this->inventarios = MovimientoInventarioExistencia::whereHas('almacen', function ($q) {
                $q->where('tipo_almacen', 'GENERAL');
            })
                ->with('almacen.sucursal:idbranchoffice,razon_social')
                ->where('idproducto', $producto->idproduct)
                ->get()
                ->toArray();
        }
        $this->generateBarcode();
        $this->calcularTrasladoDesdePrecioBase();
        $this->calcularPreciosVentaDesdeBase();
        $this->calcularVentasUtilidades();
    }
    public function render()
    {
        $unidades = [];
        if (!StringUtils::isEmpty($this->search['unidades'])) {
            $unidades = Unidad::searchOr(['clave_unidad', 'nombre', 'descripcion'], $this->search['unidades'])
                ->paginate(15);
        }

        $productServices = [];
        if (!StringUtils::isEmpty($this->search['productos_servicios'])) {
            $productServices = ProductoServicio::searchOr(['clave_prodserv', 'descripcion'], $this->search['productos_servicios'])
                ->paginate(15);
        }

        $compCriterio = $this->search['componente_criterio'];
        $searchProductComponents = [];
        if (!StringUtils::isEmpty($this->search['componentes'])) {
            if ($compCriterio !== 'auto') {
                $searchProductComponents = Producto::searchOr(['descripcion', 'clave', 'codigodebarras'], $this->search['componentes'])
                    ->where('esproductocompuesto', true)
                    ->paginate(15);
            } else {
                $searchProductComponents = Producto::search($compCriterio, $this->search['componentes'])
                    ->where('esproductocompuesto', false)
                    ->paginate(15);
            }
        }

        $historicoSalidaQuery = null;
        if (session('salidaHistoricoSearching', false)) {
            if ($this->activeTab === 'h-salidas') {
                $historicoSalidaQuery = $this->getHistoricGenericQuery(-1);
            } else {
                session(['salidaHistoricoSearching' => false]);
            }
        }
        $historicoSalidaItems = $historicoSalidaQuery?->lazyById();
        $historicoEntradaQuery = null;
        if (session('entradaHistoricoSearching', false)) {
            if ($this->activeTab === 'h-entradas') {
                $historicoEntradaQuery = $this->getHistoricGenericQuery(1);
            } else {
                session(['entradaHistoricoSearching' => false]);
            }
        }
        $historicoEntradaItems = $historicoEntradaQuery?->lazyById();

        return view('livewire.catalogos.productos.form', [
            'unidades' => $unidades,
            'productServices' => $productServices,
            'searchProductComponents' => $searchProductComponents,
            'historicoSalidaItems' => $historicoSalidaItems,
            'historicoEntradaItems' => $historicoEntradaItems
        ]);
    }

    public function save()
    {
        DB::beginTransaction();

        try {
            $updatedProduct = Producto::where('idproduct', $this->idproduct)->firstOrFail();

            if ($updatedProduct->clave != $this->form['clave']) {
                $this->validateClaveValue($this->form['clave']);
            }

            $imagen = $this->form['imagen'];
            $form['imagen'] = null;
            $updatedProduct->fill($this->form);

            if ($imagen !== $updatedProduct->getOriginal('imagen')) {
                $updatedProduct->imagen = $this->saveProductImage($imagen, $updatedProduct->idproduct);
            } else {
                $updatedProduct->imagen = $imagen;
            }
            /* ====================================================
        * POR EL MOMENTO EL IDIMPUESTO E IDUNIDADMEDIDA SE ASIGNA DESDE WEB
        * MÁS TARDE SE ASIGNARÁ DESDE ESCRITORIO TANTO PARA BASES EXISTENTES
        * COMO PARA BASES PARA INSTALACIONES NUEVAS
        * =====================================================
        */
            $idimpuesto = 3;
            $idunidadmedida = 1;
            $updatedProduct->idimpuesto = $idimpuesto;
            $updatedProduct->idunidadmedida = $idunidadmedida;

            if (!$this->cambiarCostosPromedio) {
                //Restableciendo los valores originales de costos promedio
                $updatedProduct->costo_promedio_antesiva = $updatedProduct->getOriginal('costo_promedio_antes_iva');
                $updatedProduct->costo_promedio_iva = $updatedProduct->getOriginal('costo_promedio_iva');
                $updatedProduct->costo_promedio_despuesiva = $updatedProduct->getOriginal('costo_promedio_despuesiva');
            }

            $updatedProduct->origen_registro = 'WEB';

            if ($updatedProduct->esproductocompuesto) {
                $this->updateProductoCompuestos($updatedProduct->idproduct);
            }

            if ($updatedProduct->manejainventario) {
                $almacenes = $this->warehouses;
                if (isset($this->inventarios) && count($this->inventarios) > 0) {
                    $almacenes = array_map(function ($inv) {
                        return $inv['almacen'];
                    }, $this->inventarios);
                }
                $this->updateInventarios($updatedProduct, $almacenes);
            }

            $updatedProduct->save();

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }

        DB::commit();
        $this->livewireAlert
        ->success()
        ->text('Producto actualizado correctamente')
        ->onDismiss('dismiss')
        ->show();
    }

    public function dismiss()
    {
        redirect()->route('d-catalogos-productos');
    }

    private function updateInventarios($product, $almacenes)
    {
        foreach ($almacenes as $almacen) {
            $vConcepto = '';
            $accion = $almacen['accion'];
            if ($accion == 'ENTRADA') {
                $vConcepto = 'ENTRADA POR INVENTARIOS';
            } elseif ($accion == 'SALIDA') {
                $vConcepto = 'SALIDA POR INVENTARIOS';
            } else { /*INICIAL*/
                $vConcepto = 'ESTABLECER INVENTARIOS';
            }

            $conceptoInv = Concepto::firstOrFail();

            $currentDate = Carbon::now();
            $currentUser = Auth::user();
            $idSucursal = $almacen['idsucursal'];
            $idAlmacen = $almacen['idwarehouse'];
            $cantidadAjuste = $almacen['cantidadajuste'];

            $movInvGral = MovimientoInventarioGeneral::create([
                'idsucursal' => $idSucursal,
                'idinventariofisicogral' => 0,
                'fecha' => $currentDate->toDateString(),
                'idalmacen' => $idAlmacen,
                'estatus' => Status::CERRADA->value,
                'idconceptoinventario' => $conceptoInv->idinventoryconcept,
                'idusuario' => $currentUser->id,
                'costoinventario' => $product->precioultimacompra * $cantidadAjuste,
                'idalmacendestino' => 0, /*No hay destino; es movimiento desde catálogo*/
                'idorigeninventario' => 3, /*contiene el cat_origeninventario.idinventorysource*/
                'comentarios' => '(TIPO MOVIMIENTO=INVENTARIO) - MOVIMIENTO CORRESPONDE A ' . $conceptoInv->conceptoinventario,
                'horamovimiento' => $currentDate->format('H:i:s'),
                'nombre_proveedor' => '', //No hay proveedor; es un movimiento desde catálogo
                'idturno' => 0, /*No hay turno en web*/
                'fechahora_registro' => $currentDate,
                'usuario' => $currentUser->nombre . '-WEB',
                'from_host' => 'SERVIDOR WEB',
                'origen_movimiento' => $conceptoInv->conceptoinventario,
                'tipo_movimiento' => $accion,
                'signo' => $conceptoInv->signo,
                'origen_registro' => OrigenRegistro::WEB->value,
                'descargado' => 0,
                'idmovinventariogral_desk' => 0
            ]);

            $marcaClave = Marca::find($product->idmarca)->clave;
            $familiaClave = Familia::find($product->idfamilia)->clave;

            $movDet = new MovimientoInventarioDetalle;

            $movDet->idsucursal = $idSucursal;
            $movDet->idinventariofisicodet = 0;
            $movDet->idinventariofisicogral = $movInvGral->idmovinventariogral;
            $movDet->idproducto = $product->idproduct;
            $movDet->justificacion = $vConcepto . ' (CATALOGO DE PRODUCTOS)';
            $movDet->costoinventario = $product->precioultimacompra * $cantidadAjuste;
            $movDet->idsucursal_local = $idSucursal;
            $movDet->cantidad = $cantidadAjuste;
            $movDet->folio_lote_entrada = 0;
            $movDet->lote_entrada = 0;
            $movDet->idalmacen = $idAlmacen;

            /*=======================================
                * INDEFINIDOS POR EL MOMENTO
            * =======================================
            */
            $movDet->existencia = 0;
            $movDet->fisico = 0;
            $movDet->faltante = 0;
            $movDet->excedente = 0;
            $movDet->idsucursal_web_origen  = 0;
            $movDet->idsucursal_web_destino = 0;
            /*======================================*/

            $movDet->origen_registro = OrigenRegistro::WEB->value;
            $movDet->descargado = 0;


            $movDet->idfamilia = $product->idfamilia;
            $movDet->idmarca = $product->idmarca;
            $movDet->fecha_hora = $currentDate;
            $movDet->nombre_corto_producto = strtoupper(trim($product->nombrecorto));
            $movDet->clave_familia = $familiaClave;
            $movDet->clave_marca = $marcaClave;

            $movDet->idconceptoinventario = $conceptoInv->idinventoryconcept;

            $movDet->save();
        }
    }
    private function updateProductoCompuestos($idproduct)
    {
        foreach ($this->productComponents as $newProductComponent) {
            if (!isset($newProductComponent['idcompoundproduct'])) {
                //Crear un nuevo producto componente
                ProductoCompuesto::create([
                    'idcomponente' => $newProductComponent['idcomponente'],
                    'idproducto' => $idproduct,
                    'cantidad' => $newProductComponent['cantidad'],
                    'origen_registro' => OrigenRegistro::WEB->value
                ]);
            }
        }

        $removedIds = array_map(function ($removedProductComponent) {
            if (isset($removedProductComponent->value['idcompoundproduct'])) {
                return $removedProductComponent->value['idcompoundproduct'];
            }
        }, $this->removedProductComponents);

        ProductoCompuesto::destroy($removedIds);
    }

    public function return()
    {
        redirect()->route('d-catalogos-productos');
    }

    public function getHistoricoEntradas()
    {
        session(['entradaHistoricoSearching' => true]);
        $this->resetPage();
        $this->updateHistoricGenericResume('historicoEntrada', 1);
    }

    public function getHistoricoSalidas()
    {
        session(['salidaHistoricoSearching' => true]);
        $this->resetPage();
        $this->updateHistoricGenericResume('historicoSalida', -1);
    }

    private function getHistoricGenericQuery(int $signo)
    {
        return BitacoraExistencia::with('sucursal:idbranchoffice,razon_social')
            ->where('signo', $signo)
            ->where('idproducto', $this->idproduct);
    }

    private function updateHistoricGenericResume($model, $signo)
    {
        $query = $this->getHistoricGenericQuery($signo);
        $this->{$model}['total'] = (clone $query)->count();
        $this->{$model}['sumatoria'] = (clone $query)->sum('movimiento_cantidad');
    }
}
