<?php

namespace App\Livewire\Catalogos\Productos;

use App\Enums\OrigenRegistro;
use App\Enums\Status;
use App\Models\Almacen;
use App\Models\Concepto;
use App\Models\Familia;
use App\Models\Marca;
use App\Models\MovimientoInventarioDetalle;
use App\Models\MovimientoInventarioGeneral;
use App\Models\Producto;
use App\Models\ProductoCompuesto;
use App\Models\ProductoServicio;
use App\Models\Unidad;
use App\Repositories\Product\ProductRepositoryInterface;
use App\Support\StringUtils;
use App\Traits\WithLivewireAlert;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;

#[Title('Nuevo producto')]
class Create extends Component
{
    use WithFileUploads;
    use WithPagination;
    use ProductBaseLogic;
    use WithLivewireAlert;

    public $families;
    public $brands;
    public $warehouses;
    public $productComponents;

    public $action = 'create';

    public function mount()
    {
        $this->families = Familia::all();
        $defaultFamily = Familia::where('clave', 'SF')->first();
        $this->form['idfamilia'] = $defaultFamily->idfamilia ?? '';

        $this->brands = Marca::all();
        $defaultBrand = Marca::where('clave', 'SINMARCA')->first();
        $this->form['idmarca'] = $defaultBrand->idmarca ?? '';

        $this->warehouses = Almacen::with('sucursal')->get()->toArray();

        $defaultProdServ = ProductoServicio::where('idprodserv', 1)->first();
        $this->form['clave_prodserv'] = $defaultProdServ->clave_prodserv ?? '';
        $this->form['prodserv'] = $defaultProdServ->descripcion ?? '';

        $defaultUnit = Unidad::where('clave_unidad', 'H87')->first();
        $this->form['clave_unidadmedida'] = $defaultUnit->clave_unidad ?? '';
        $this->form['unidadmedida'] = $defaultUnit->nombre ?? '';

        $this->form['fechaultimacompra'] = date('Y-m-d');
        $this->productComponents = [];
    }
    public function render()
    {
        $unidades = [];
        if (!StringUtils::isEmpty($this->search['unidades'])) {
            $unidades = Unidad::searchOr(['clave_unidad', 'nombre', 'descripcion'], $this->search['unidades'])
                ->paginate(15);
        }

        $productServices = [];
        if (!StringUtils::isEmpty($this->search['productos_servicios'])) {
            $productServices = ProductoServicio::searchOr(['clave_prodserv', 'descripcion'], $this->search['productos_servicios'])
                ->paginate(15);
        }
        $compCriterio = $this->search['componente_criterio'];
        $searchProductComponents = [];
        if (!StringUtils::isEmpty($this->search['componentes'])) {
            if ($compCriterio !== 'auto') {
                $searchProductComponents = Producto::search($compCriterio, $this->search['componentes'])
                    ->where('esproductocompuesto', false)
                    ->paginate(15);
            } else {
                $searchProductComponents = Producto::searchOr(['descripcion', 'clave', 'codigodebarras'], $this->search['componentes'])
                    ->where('esproductocompuesto', true)
                    ->paginate(15);
            }
        }

        return view('livewire.catalogos.productos.form', [
            'unidades' => $unidades,
            'productServices' => $productServices,
            'searchProductComponents' => $searchProductComponents
        ]);
    }

    public function return()
    {
        redirect()->route('d-catalogos-productos');
    }

    public function save()
    {
        try {
            $this->validateClaveValue($this->form['clave']);
            $this->validate(attributes: $this->atributtes);
        } catch (ValidationException $e) {
            $this->livewireAlert
                ->error()
                ->text('Algunos campos no se han completado de manera correcta. Verifique el valor de los campos.')
                ->withConfirmButton('Aceptar')
                ->show();
            throw $e;
        }
        $this->form['fechaalta'] = date('Y-m-d');
        $this->form['origen_registro'] = OrigenRegistro::WEB->value;
        $image = $this->form['imagen'];
        $this->form['imagen'] = null;
        $productImageUrl = null;
        try {
            DB::beginTransaction();
            $newProduct = Producto::create($this->form);
            $idProduct = $newProduct->idproduct;
            if ($image) {
                $productImageUrl = $this->saveProductImage($image, $idProduct);
                if (!$productImageUrl) {
                    throw new \Exception('Error al guardar la imagen del producto.');
                }
                $newProduct->imagen = $productImageUrl;
                $newProduct->save();
            }

            if ($newProduct->esproductocompuesto) {
                $this->saveProductComponents($idProduct);
            }

            if ($newProduct->manejainventario) {
                $this->saveInventarios($newProduct);
            }
            $newProduct->idproducto = $idProduct;
            $newProduct->save();

        } catch (\Exception $e) {
            DB::rollBack();
            if ($productImageUrl) {
                unlink(storage_path('app/' . $productImageUrl));
                $this->form['imagen'] = $image;
            }
            logger()->error('Error al crear el producto: ' . $e->getMessage());
            $this->livewireAlert
                ->error()
                ->title('¡Error inesperado!')
                ->text('Intentelo más tarde')
                ->show();
            return;
        }

        DB::commit();

        $this->livewireAlert
            ->success()
            ->text('Se ha creado el producto ' . $this->form['nombrecorto'] . ' exitosamente')
            ->withConfirmButton('Aceptar')
            ->onConfirm('return')
            ->show();
    }

    private function saveInventarios(Producto $product)
    {
        $conceptoInv = Concepto::where('conceptoinventario', 'ENTRADA POR INVENTARIOS')->firstOrFail();
        $currentUser = Auth::user();
        
        foreach ($this->warehouses as $warehouse) {
            if ($warehouse['cantidadajuste'] <= 0) {
                continue;
            }
            $currentDate = Carbon::now();
            $idAlmacen = $warehouse['idwarehouse'];
            $idSucursal = $warehouse['sucursal']['idbranchoffice'];
            $cantidadAjuste = $warehouse['cantidadajuste'];

            $movInvGral = MovimientoInventarioGeneral::create([
                'idsucursal' => $idSucursal,
                'idinventariofisicogral' => 0,
                'fecha' => $currentDate->toDateString(),
                'idalmacen' => $idAlmacen,
                'estatus' => Status::CERRADA->value,
                'idconceptoinventario' => $conceptoInv->idinventoryconcept,
                'idusuario' => $currentUser->id,
                'costoinventario' => $product->precioultimacompra * $cantidadAjuste,
                'idalmacendestino' => 0, /*No hay destino; es movimiento desde catálogo*/
                'idorigeninventario' => 3, /*contiene el cat_origeninventario.idinventorysource*/
                'comentarios' => '(TIPO MOVIMIENTO=INVENTARIO) - MOVIMIENTO CORRESPONDE A ' . $conceptoInv->conceptoinventario,
                'horamovimiento' => $currentDate->format('H:i:s'),
                'nombre_proveedor' => '', //No hay proveedor; es un movimiento desde catálogo
                'idturno' => 0, /*No hay turno en web*/
                'fechahora_registro' => $currentDate,
                'usuario' => $currentUser->nombre . '-WEB',
                'from_host' => 'SERVIDOR WEB',
                'origen_movimiento' => $conceptoInv->conceptoinventario,
                'tipo_movimiento' => 'ENTRADA',
                'signo' => 1, /*Es una entrada, el sigo es positivo*/
                'origen_registro' => OrigenRegistro::WEB->value,
                'descargado' => 0,
                'idmovinventariogral_desk' => 0
            ]);

            $marcaClave = Marca::find($product->idmarca)->clave;
            $familiaClave = Familia::find($product->idfamilia)->clave;

            $movDet = new MovimientoInventarioDetalle;

            $movDet->idsucursal = $idSucursal;
            $movDet->idinventariofisicodet = 0;
            $movDet->idinventariofisicogral = $movInvGral->idmovinventariogral;
            $movDet->idproducto = $product->idproduct;
            $movDet->justificacion = 'ENTRADA POR INVENTARIOS (CATALOGO DE PRODUCTOS)';
            $movDet->costoinventario = $product->precioultimacompra * $cantidadAjuste;
            $movDet->idsucursal_local = $idSucursal;
            $movDet->cantidad = $cantidadAjuste;
            $movDet->folio_lote_entrada = 0;
            $movDet->lote_entrada = 0;
            $movDet->idalmacen = $idAlmacen;

            /*=======================================
                * INDEFINIDOS POR EL MOMENTO
            * =======================================
            */
            $movDet->existencia = 0;
            $movDet->fisico = 0;
            $movDet->faltante = 0;
            $movDet->excedente = 0;
            $movDet->idsucursal_web_origen  = 0;
            $movDet->idsucursal_web_destino = 0;
            /*======================================*/

            $movDet->origen_registro = OrigenRegistro::WEB->value;
            $movDet->descargado = 0;


            $movDet->idfamilia = $product->idfamilia;
            $movDet->idmarca = $product->idmarca;
            $movDet->fecha_hora = $currentDate;
            $movDet->nombre_corto_producto = strtoupper(trim($product->nombrecorto));
            $movDet->clave_familia = $familiaClave;
            $movDet->clave_marca = $marcaClave;

            $movDet->idconceptoinventario = $conceptoInv->idinventoryconcept;

            $movDet->save();
            dump('Detalle de inventario guardado con ID: ' . $movDet->idmovinvdet);
        }
    }

    private function saveProductComponents($idProduct)
    {
        foreach ($this->productComponents as $productComponent) {
            ProductoCompuesto::create([
                'idcomponente' => $productComponent['idcomponente'],
                'idproducto' => $idProduct,
                'cantidad' => $productComponent['cantidad'],
                'origen_registro' => OrigenRegistro::WEB->value
            ]);
        }
    }
}
