<?php

namespace App\Http\Requests\Inventarios;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class RegistrarTraspasoRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'partidas' => 'required|array|min:1|max:300',
            'partidas.*.producto_id' => 'required|exists:cat_productos,idproduct',
            'partidas.*.cantidad' => 'required|numeric|min:0.001',
            'partidas.*.precio' => 'nullable|numeric|min:0',
            'partidas.*.importe' => 'required|numeric|min:0',
            'partidas.*.lote_origen' => 'nullable|string|max:50',
            'partidas.*.fecha_caducidad' => 'nullable|date|after:yesterday',
            'partidas.*.justificacion' => 'nullable|string|max:255',
            'almacen_origen_id' => 'required|exists:cat_almacenes,idwarehouse',
            'almacen_destino_id' => 'required|exists:cat_almacenes,idwarehouse|different:almacen_origen_id',
            'importe_total' => 'required|numeric|min:0',
            'proveedor_id' => 'nullable|exists:cat_proveedores,idprovider',
            'fecha_movimiento' => 'required|date|before_or_equal:today',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'partidas.required' => 'Debe agregar al menos una partida para el traspaso.',
            'partidas.min' => 'Debe agregar al menos una partida para el traspaso.',
            'partidas.max' => 'No puede agregar más de 300 partidas por traspaso.',
            'partidas.*.producto_id.required' => 'Cada partida debe tener un producto asociado.',
            'partidas.*.producto_id.exists' => 'El producto seleccionado no existe.',
            'partidas.*.cantidad.required' => 'La cantidad es obligatoria para cada partida.',
            'partidas.*.cantidad.min' => 'La cantidad debe ser mayor a 0.',
            'partidas.*.importe.required' => 'El importe es obligatorio para cada partida.',
            'almacen_origen_id.required' => 'Debe seleccionar un almacén de origen.',
            'almacen_origen_id.exists' => 'El almacén de origen seleccionado no existe.',
            'almacen_destino_id.required' => 'Debe seleccionar un almacén de destino.',
            'almacen_destino_id.exists' => 'El almacén de destino seleccionado no existe.',
            'almacen_destino_id.different' => 'El almacén de destino debe ser diferente al almacén de origen.',
            'importe_total.required' => 'El importe total es obligatorio.',
            'fecha_movimiento.required' => 'La fecha del movimiento es obligatoria.',
            'fecha_movimiento.before_or_equal' => 'La fecha del movimiento no puede ser futura.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $validator) {
            // Validar que las sucursales sean diferentes
            $almacenOrigen = \App\Models\Almacen::find($this->input('almacen_origen_id'));
            $almacenDestino = \App\Models\Almacen::find($this->input('almacen_destino_id'));

            if ($almacenOrigen && $almacenDestino) {
                if ($almacenOrigen->idsucursal === $almacenDestino->idsucursal) {
                    $validator->errors()->add(
                        'almacen_destino_id',
                        'El traspaso debe realizarse entre sucursales diferentes.'
                    );
                }
            }

            // Validar existencias disponibles
            $partidas = $this->input('partidas', []);
            $almacenOrigenId = $this->input('almacen_origen_id');

            foreach ($partidas as $index => $partida) {
                $inventario = \Illuminate\Support\Facades\DB::table('mov_inv_existencias')
                    ->where('idalmacen', $almacenOrigenId)
                    ->where('idproducto', $partida['producto_id'])
                    ->first();

                if (!$inventario) {
                    $validator->errors()->add(
                        "partidas.{$index}.producto_id",
                        'El producto no tiene existencias en el almacén de origen.'
                    );
                    continue;
                }

                if ($inventario->existencia < $partida['cantidad']) {
                    $producto = \App\Models\Producto::find($partida['producto_id']);
                    $nombreProducto = $producto ? $producto->nombrecorto : 'el producto';
                    $validator->errors()->add(
                        "partidas.{$index}.cantidad",
                        "Stock insuficiente para {$nombreProducto}. Disponible: {$inventario->existencia}"
                    );
                }
            }
        });
    }
}
