<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Repositories\PuntoDeVenta\PuntoDeVentaInterface;
use Illuminate\Http\Request;
use OpenApi\Attributes as OA;

#[OA\Tag(name: 'Punto de Venta', description: 'Gestión de ventas en el punto de venta')]
class PuntoVentaController extends Controller
{
    public function __construct(private readonly PuntoDeVentaInterface $puntoDeVenta) {}

    #[OA\Get(
        path: '/api/punto-venta',
        summary: 'Listar ventas',
        description: 'Obtiene el listado de ventas del punto de venta con filtros opcionales',
        security: [['bearerAuth' => []]],
        tags: ['Punto de Venta'],
        parameters: [
            new OA\Parameter(
                name: 'fecha_inicio',
                description: 'Fecha de inicio del filtro',
                in: 'query',
                required: false,
                schema: new OA\Schema(type: 'string', format: 'date')
            ),
            new OA\Parameter(
                name: 'fecha_fin',
                description: 'Fecha de fin del filtro',
                in: 'query',
                required: false,
                schema: new OA\Schema(type: 'string', format: 'date')
            ),
            new OA\Parameter(
                name: 'cliente_id',
                description: 'ID del cliente',
                in: 'query',
                required: false,
                schema: new OA\Schema(type: 'integer')
            ),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Listado de ventas obtenido exitosamente',
                content: new OA\JsonContent(
                    type: 'array',
                    items: new OA\Items(
                        properties: [
                            new OA\Property(property: 'id', type: 'integer', example: 1),
                            new OA\Property(property: 'folio', type: 'string', example: 'VTA-001'),
                            new OA\Property(property: 'total', type: 'number', format: 'float', example: 1500.50),
                            new OA\Property(property: 'fecha', type: 'string', format: 'datetime'),
                            new OA\Property(property: 'cliente', type: 'object'),
                        ]
                    )
                )
            ),
        ]
    )]
    public function index(Request $request)
    {
        $filters = $request->all();
    $result = $this->puntoDeVenta->obtenerVentas($filters);
        return response()->json($result);
    }

    #[OA\Post(
        path: '/api/punto-venta',
        summary: 'Registrar una nueva venta',
        description: 'Crea una nueva venta en el punto de venta',
        security: [['bearerAuth' => []]],
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\JsonContent(
                required: ['productos', 'total'],
                properties: [
                    new OA\Property(property: 'cliente_id', type: 'integer', example: 1),
                    new OA\Property(
                        property: 'productos',
                        type: 'array',
                        items: new OA\Items(
                            properties: [
                                new OA\Property(property: 'producto_id', type: 'integer', example: 5),
                                new OA\Property(property: 'cantidad', type: 'number', example: 2),
                                new OA\Property(property: 'precio', type: 'number', format: 'float', example: 150.00),
                            ]
                        )
                    ),
                    new OA\Property(property: 'subtotal', type: 'number', format: 'float', example: 1300.00),
                    new OA\Property(property: 'descuento', type: 'number', format: 'float', example: 0.00),
                    new OA\Property(property: 'iva', type: 'number', format: 'float', example: 200.50),
                    new OA\Property(property: 'total', type: 'number', format: 'float', example: 1500.50),
                    new OA\Property(property: 'metodo_pago', type: 'string', example: 'efectivo'),
                    new OA\Property(property: 'observaciones', type: 'string', example: 'Venta con descuento'),
                ]
            )
        ),
        tags: ['Punto de Venta'],
        responses: [
            new OA\Response(
                response: 201,
                description: 'Venta creada exitosamente',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'id', type: 'integer', example: 1),
                        new OA\Property(property: 'folio', type: 'string', example: 'VTA-001'),
                        new OA\Property(property: 'total', type: 'number', format: 'float', example: 1500.50),
                        new OA\Property(property: 'message', type: 'string', example: 'Venta registrada exitosamente'),
                    ]
                )
            ),
            new OA\Response(
                response: 400,
                description: 'Datos inválidos',
            ),
        ]
    )]
    public function store(Request $request)
    {
        $data = $request->all();
        if (empty($data)) {
            $raw = $request->getContent();
            if (is_string($raw) && $raw !== '') {
                $decoded = json_decode($raw, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $data = $decoded;
                }
            }
        }
        $result = $this->puntoDeVenta->crearVenta($data);
        return response()->json($result, 201);
    }

    #[OA\Get(
        path: '/api/punto-venta/{id}',
        summary: 'Obtener detalles de una venta',
        description: 'Obtiene los detalles completos de una venta específica',
        security: [['bearerAuth' => []]],
        tags: ['Punto de Venta'],
        parameters: [
            new OA\Parameter(
                name: 'id',
                description: 'ID de la venta',
                in: 'path',
                required: true,
                schema: new OA\Schema(type: 'integer')
            ),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Detalles de la venta',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'id', type: 'integer', example: 1),
                        new OA\Property(property: 'folio', type: 'string', example: 'VTA-001'),
                        new OA\Property(property: 'total', type: 'number', format: 'float', example: 1500.50),
                        new OA\Property(property: 'fecha', type: 'string', format: 'datetime'),
                        new OA\Property(property: 'cliente', type: 'object'),
                        new OA\Property(property: 'productos', type: 'array', items: new OA\Items(type: 'object')),
                    ]
                )
            ),
            new OA\Response(
                response: 404,
                description: 'Venta no encontrada',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'message', type: 'string', example: 'Venta no encontrada'),
                    ]
                )
            ),
        ]
    )]
    public function show(int $id)
    {
        $venta = $this->puntoDeVenta->obtenerVentaPorId($id);
        if (!$venta) {
            return response()->json(['message' => 'Venta no encontrada'], 404);
        }
        return response()->json($venta);
    }

    #[OA\Get(
        path: '/api/punto-venta/{id}/productos',
        summary: 'Obtener producto para el punto de venta',
        description: 'Obtiene los detalles de un producto específico para mostrar en el punto de venta',
        security: [['bearerAuth' => []]],
        tags: ['Punto de Venta'],
        parameters: [
            new OA\Parameter(
                name: 'id',
                description: 'ID del producto',
                in: 'path',
                required: true,
                schema: new OA\Schema(type: 'integer')
            ),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Detalles del producto',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'id', type: 'integer', example: 5),
                        new OA\Property(property: 'codigo', type: 'string', example: 'PROD-001'),
                        new OA\Property(property: 'nombre', type: 'string', example: 'Producto ejemplo'),
                        new OA\Property(property: 'precio', type: 'number', format: 'float', example: 150.00),
                        new OA\Property(property: 'stock', type: 'integer', example: 50),
                    ]
                )
            ),
            new OA\Response(
                response: 404,
                description: 'Producto no encontrado',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'message', type: 'string', example: 'Producto no encontrado'),
                    ]
                )
            ),
        ]
    )]
    public function showProductos(int $id)
    {
        $producto = $this->puntoDeVenta->obtenerProductos(['id' => $id]);
        if (!$producto) {
            return response()->json(['message' => 'Producto no encontrado'], 404);
        }
        return response()->json($producto);
    }
}
