<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PedidosIntermediosController extends Controller
{
    // \Log::debug("pedido_intermedio_general: " . count($request->pedido_intermedio_general));
    public function store(Request $request)
    {
        try
        {
            $this->validarParametros($request);
        }
        catch(Exception $e)
        {
            return response()->json([
                'status' => false,
                'errors' => [$e->getMessage()]
            ], 400);
        }

        $idSucursal      = $request->idsucursal;
        $infoGral        = $request->pedido_intermedio_general;
        $arrInfoDetalles = $request->pedido_intermedio_detalles;

        // En sistema tenant, no necesitamos idempresa
        $config = $this->ObtenerConfiguracion();

        $oDataResponse = new \stdClass;

        DB::transaction(function () use($infoGral, $arrInfoDetalles, $idSucursal, $oDataResponse, $config) {

            $arrPedInterGralInsert = $this->arrayEliminarElementosConValorNull($infoGral);
            $arrPedInterGralInsert['idsucursal'] = $idSucursal;
            $arrPedInterGralInsert['idempresa'] = tenant('id'); // Agregar tenant ID como idempresa

            // No agregar afectar_inventarios a la inserción, solo se usa para la lógica
            // $config->afectainventarios se usa más adelante en procesarInventarios()

            $idPedidoIntermedioGral = DB::table('pedidos_intermedios_general_movil')
                ->insertGetId($arrPedInterGralInsert);

            $oDataResponse->pedido_intermedio_general = new \stdClass;
            $oDataResponse->pedido_intermedio_general->idweb = $idPedidoIntermedioGral;
            $oDataResponse->pedidos_intermedios_detalle = [];

            $iTotalDet = count($arrInfoDetalles);

            for ($i=0; $i < $iTotalDet; $i++){
                $arrDetInsert = $this->arrayEliminarElementosConValorNull($arrInfoDetalles[$i]);

                $arrDetInsert['idsucursal']                         = $idSucursal;
                $arrDetInsert['idempresa']                          = tenant('id'); // Agregar tenant ID como idempresa
                $arrDetInsert['id_pedido_intermedio_general_movil'] = $idPedidoIntermedioGral;

                $iIDPedDet = DB::table('pedidos_intermedios_detalle_movil')->insertGetId($arrDetInsert);
                
                // Agregar el ID recién insertado al array para la lógica de inventarios
                $arrDetInsert['id'] = $iIDPedDet;
                
                // Aplicar lógica del trigger para manejo de inventarios
                $this->procesarInventarios($arrDetInsert, $idPedidoIntermedioGral, $infoGral, $config->afectainventarios);
                
                $oDataResponse->pedidos_intermedios_detalle[$i] = new \stdClass;
                $oDataResponse->pedidos_intermedios_detalle[$i]->idweb = $iIDPedDet;
            }
        });

        return response()->json([
            'status' => true,
            'message' => 'Registro exitoso.',
            'data' => $oDataResponse,
        ], 200);
    }

    public function show($id)
    {
        if(! isset($id))
        {
            return response()->json([
                'status' => false,
                'errors' => ['No se especificó la propiedad "id"']
            ], 400);
        }

        $oPedidoIntermedioGral = DB::table('pedidos_intermedios_general_movil')
            ->select('id', 'idempresa', 'idsucursal', 'idstatus', 'fecha_pedido',
                'fecha_pedido_entrega', 'idcliente', 'nombre_cliente', 'idvendedor',
                'nombre_vendedor', 'pedidoyaaplicado', 'comentarios')
            ->where('id', $id)
            ->where('idempresa', tenant('id')) // Filtrar por tenant
            ->first();

        if($oPedidoIntermedioGral == null){
            return response()->json([
                'status' => false,
                'message' => ['No se encontraron registros']
            ], 200);
        }

        $oPedidoIntemedioDet = DB::table('pedidos_intermedios_detalle_movil')
            ->select('id', 'idempresa', 'idsucursal', 'id_pedido_intermedio_general_movil',
            'idproducto', 'clave', 'descripcion', 'unidadmedida', 'cantidad', 'cantidadyaenviada',
            'cantidadxenviar', 'precio', 'importe', 'descuentox100', 'descuento', 'subtotal',
            'iva', 'total', 'traslado_iva_tipo_factor', 'traslado_iva_tasa_cuota',
            'traslado_iva_importe', 'traslado_ieps_tipo_factor', 'traslado_ieps_tasa_cuota',
            'traslado_ieps_importe', 'retencion_iva_tipo_factor', 'retencion_iva_tasa_cuota',
            'retencion_iva_importe', 'retencion_isr_tipo_factor', 'retencion_isr_tasa_cuota',
            'retencion_isr_importe')
            ->where('id_pedido_intermedio_general_movil', $oPedidoIntermedioGral->id)
            ->where('idempresa', tenant('id')) // Filtrar por tenant
            ->get();

        return response()->json([
            'status' => true,
            'data' => [
                'pedido_intermedio_general' => $oPedidoIntermedioGral,
                'pedidos_intermedios_detalle' => $oPedidoIntemedioDet
            ],
            'message' => 'Información de pedidos intermedios general y detalles.',
        ], 200);
    }

    /**
     * Devuelve un array especificado, eliminando los elementos que contengan valores null.
     * @date 2023-12-24
     * @author RRPC
     * @param array $arrColumnas
     * @return array $arrSoloColumnasConValor
     */
    private function arrayEliminarElementosConValorNull($arrColumnas)
    {
        $arrSoloColumnasConValor = [];

        foreach ($arrColumnas as $key => $value) {
            if(isset($value)){
                $arrSoloColumnasConValor[$key] = $value;
            }
        }

        return $arrSoloColumnasConValor;
    }

    /**
     * Valida los parámetros obligatorios y que cumplan con cierto criterio.
     * Si falla una validación, lanza una excepción.
     * @date 2023-12-31
     * @author Ricardo Pacab
     * @param mixed $request
     */
    private function validarParametros($request)
    {
        if (empty($request->json()->all())) {
            throw new Exception('La solicitud no contiene un JSON válido.');
        }

        if(! isset($request->idsucursal)){
            throw new Exception('No se especificó la propiedad "idsucursal".');
        }

        if(! isset($request->pedido_intermedio_general)){
            throw new Exception('No se especificó la propiedad "pedido_intermedio_general".');
        }

        if(!isset($request->pedido_intermedio_detalles) || !count($request->pedido_intermedio_detalles) > 0){
            throw new Exception('Se requiere al menos un elemento en la propiedad "pedido_intermedio_detalles".');
        }
    }

    /**
     * Obtiene la configuración de pedidos intermedios.
     * @date 2025-08-24
     * @author Ricardo Pacab
     * @return instancia de DB de la tabla 'config'.
     */
    private function ObtenerConfiguracion()
    {
        // En sistema tenant, usar el tenant_id desde la configuración de tenancy
        $tenantId = tenant('id');

        // Debug: Log para verificar el tenant_id
        Log::info('Tenant actual: ' . ($tenantId ?? 'null'));

        $config = DB::connection('mysql')->table('config')
        ->select('afectainventarios', 'habilitarprecio')
        ->where('idempresa', $tenantId)
        ->first();

        // Si no encuentra configuración, usar valores por defecto
        if (!$config) {
            Log::warning("No se encontró configuración para tenant: " . $tenantId);
            
            $config = (object) [
                'afectainventarios' => 1,
                'habilitarprecio' => 1
            ];
        }

        return $config;
    }

    /**
     * Procesa la lógica de inventarios equivalente al trigger SQL.
     * @date 2025-09-05
     * @author Marco Canche
     */

    private function procesarInventarios($detalle, $idPedidoIntermedioGral, $infoGral, $afectarInventarios)
    {
        // Variables del trigger convertidas
        $fechaHoraRegistro = $infoGral['fecha_pedido'] ?? now();
        $usuario = $infoGral['nombre_vendedor'] ?? '';
        $fromHost = 'móvil';
        $origenMovimiento = 'app-pedidos';
        $folioDocumento = $idPedidoIntermedioGral;
        $idTurno = 0;
        $signo = -1;
        $fisico = 0;
        $faltante = 0;
        $excedente = 0;

        // Obtener datos del producto
        $producto = DB::table('cat_productos')
            ->select('manejainventario', 'esproductocompuesto', 'costo_promedio_antesiva')
            ->where('idproduct', $detalle['idproducto'])
            ->first();

        if (!$producto) {
            return;
        }

        // Si maneja inventario y se debe afectar inventarios
        if ($producto->manejainventario == 1 && $afectarInventarios == 1) {
            $this->procesarInventarioSimple($detalle, $producto, $fechaHoraRegistro, $usuario, 
                $fromHost, $origenMovimiento, $folioDocumento, $idTurno, $signo, 
                $fisico, $faltante, $excedente);
        }
        // Si es producto compuesto y se debe afectar inventarios
        elseif ($producto->esproductocompuesto == 1 && $afectarInventarios == 1) {
            $this->procesarInventarioCompuesto($detalle, $fechaHoraRegistro, $usuario, 
                $fromHost, $origenMovimiento, $folioDocumento, $idTurno, $signo, 
                $fisico, $faltante, $excedente);
        }

        // Insertar en mov_changes
        DB::table('mov_changes')->insert([
            'idsucursal' => $detalle['idsucursal'],
            'identable' => $detalle['id'],
            'accion' => 'I',
            'tabla' => 'PEDIDOS_INTERMEDIOS_DETALLE_MOVIL'
        ]);
    }

    /**
     * Procesa inventario para productos simples.
     * @param array $detalle
     * @param object $producto
     * @param string $fechaHoraRegistro
     * @param string $usuario
     * @param string $fromHost
     * @param string $origenMovimiento
     * @param int $folioDocumento
     * @param int $idTurno
     * @param int $signo
     * @param float $fisico
     * @param float $faltante
     * @param float $excedente
     */
    private function procesarInventarioSimple($detalle, $producto, $fechaHoraRegistro, $usuario, 
        $fromHost, $origenMovimiento, $folioDocumento, $idTurno, $signo, 
        $fisico, $faltante, $excedente)
    {
        // Obtener existencia actual
        $existencia = DB::table('mov_inv_existencias')
            ->select('existencia')
            ->where('idproducto', $detalle['idproducto'])
            ->where('idsucursal', $detalle['idsucursal'])
            ->first();

        if ($existencia) {
            $existenciaAnterior = $existencia->existencia;
            $existenciaFinal = $existenciaAnterior - $detalle['cantidad'];

            // Insertar en bitacora_existencias
            DB::table('bitacora_existencias')->insert([
                'idsucursal' => $detalle['idsucursal'],
                'idbitacora' => $detalle['id'], // Usar el ID del detalle como idbitacora
                'fechahora_registro' => $fechaHoraRegistro,
                'tipo' => 'SALIDA X APP-PEDIDOS',
                'signo' => $signo,
                'usuario' => $usuario,
                'from_host' => $fromHost,
                'origen_movimiento' => $origenMovimiento,
                'origen_idmovimiento' => $detalle['id'],
                'origen_folio' => $folioDocumento,
                'idproducto' => $detalle['idproducto'],
                'existencia_anterior' => $existenciaAnterior,
                'movimiento_cantidad' => $detalle['cantidad'],
                'existencia_final' => $existenciaFinal,
                'idturno' => $idTurno,
                'idalmacen' => null,
                'comentarios' => '',
                'fisico' => $fisico,
                'faltante' => $faltante,
                'excedente' => $excedente,
                'precio_unitario' => $producto->costo_promedio_antesiva,
                'importe' => $detalle['cantidad'] * $producto->costo_promedio_antesiva,
                'origen_registro' => 'NUBE',
                'descargado' => 1
            ]);

            // Actualizar existencias
            DB::table('mov_inv_existencias')
                ->where('idproducto', $detalle['idproducto'])
                ->where('idsucursal', $detalle['idsucursal'])
                ->update(['existencia' => $existenciaFinal]);
        }
    }

    /**
     * Procesa inventario para productos compuestos.
     * @param array $detalle
     * @param string $fechaHoraRegistro
     * @param string $usuario
     * @param string $fromHost
     * @param string $origenMovimiento
     * @param int $folioDocumento
     * @param int $idTurno
     * @param int $signo
     * @param float $fisico
     * @param float $faltante
     * @param float $excedente
     */
    private function procesarInventarioCompuesto($detalle, $fechaHoraRegistro, $usuario, 
        $fromHost, $origenMovimiento, $folioDocumento, $idTurno, $signo, 
        $fisico, $faltante, $excedente)
    {
        // Obtener componentes del producto compuesto
        $componentes = DB::table('productoscompuestos as c')
            ->join('cat_productos as p', 'c.idcomponente', '=', 'p.idproduct')
            ->select('c.idcomponente', 'c.cantidad as cantidad_componente', 'p.manejainventario', 'p.costo_promedio_antesiva')
            ->where('c.idproducto', $detalle['idproducto'])
            ->get();

        foreach ($componentes as $componente) {
            if ($componente->manejainventario == 1) {
                // Obtener existencia del componente
                $existencia = DB::table('mov_inv_existencias')
                    ->select('existencia')
                    ->where('idproducto', $componente->idcomponente)
                    ->where('idsucursal', $detalle['idsucursal'])
                    ->first();

                if ($existencia) {
                    $existenciaAnterior = $existencia->existencia;
                    $cantidadMovimiento = $detalle['cantidad'] * $componente->cantidad_componente;
                    $existenciaFinal = $existenciaAnterior - $cantidadMovimiento;

                    // Insertar en bitacora_existencias
                    DB::table('bitacora_existencias')->insert([
                        'idsucursal' => $detalle['idsucursal'],
                        'idbitacora' => $detalle['id'], // Usar el ID del detalle como idbitacora
                        'fechahora_registro' => $fechaHoraRegistro,
                        'tipo' => 'SALIDA X APP-PEDIDOS',
                        'signo' => $signo,
                        'usuario' => $usuario,
                        'from_host' => $fromHost,
                        'origen_movimiento' => $origenMovimiento,
                        'origen_idmovimiento' => $detalle['id'],
                        'origen_folio' => $folioDocumento,
                        'idproducto' => $componente->idcomponente,
                        'existencia_anterior' => $existenciaAnterior,
                        'movimiento_cantidad' => $cantidadMovimiento,
                        'existencia_final' => $existenciaFinal,
                        'idturno' => $idTurno,
                        'idalmacen' => null,
                        'comentarios' => 'SALIÓ COMO COMPONENTE DE ' . ($detalle['descripcion'] ?? ''),
                        'fisico' => $fisico,
                        'faltante' => $faltante,
                        'excedente' => $excedente,
                        'precio_unitario' => $componente->costo_promedio_antesiva,
                        'importe' => $cantidadMovimiento * $componente->costo_promedio_antesiva,
                        'origen_registro' => 'NUBE',
                        'descargado' => 1
                    ]);

                    // Actualizar existencias del componente
                    DB::table('mov_inv_existencias')
                        ->where('idproducto', $componente->idcomponente)
                        ->where('idsucursal', $detalle['idsucursal'])
                        ->update(['existencia' => $existenciaFinal]);
                }
            }
        }
    }
}