<?php

namespace App\Console\Commands;

use App\Models\Tenant;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class SetupTriggers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'setup-triggers
    { --path= : Ruta del archivo de configuración de triggers }
    { --rollback : Ejecuta down() (migrate:rollback) en vez de up() }
    { --step=1 : Pasos a revertir cuando usas --rollback }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $path = $this->option('path');
        $rollback = (bool) $this->option('rollback');
        $step = (int) $this->option('step');
        if (!$path) {
            $this->error('❌ Debes especificar la ruta del archivo de configuración de triggers con --path');
            return;
        }

        $this->info('Setting up triggers...');
        Tenant::query()->cursor()->each(function (Tenant $tenant) use ($path, $rollback, $step) {
            $tenant->run(function () use ($tenant, $path, $rollback, $step) {
                DB::purge('tenant');
                DB::reconnect('tenant');

                if ($rollback) {
                    $this->info("Rolling back triggers for tenant: {$tenant->id}");
                    Artisan::call('migrate:rollback', [
                        '--database' => 'tenant',
                        '--path' => $path,
                        '--step' => $step,
                        '--force' => true,
                    ]);
                } else {
                    $this->info("Migrating triggers for tenant: {$tenant->id}");
                    Artisan::call('migrate', [
                        '--database' => 'tenant',
                        '--path' => $path,
                        '--force' => true,
                    ]);
                }

                $this->output->write(Artisan::output());
                $this->info("✅ Done for tenant: {$tenant->id}");
            });
        });

        $this->info('Triggers setup completed successfully.');
    }
}
